/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.utils;

import com.android.ddmlib.IDevice;
import com.android.ddmlib.Log;
import com.android.ddmlib.Log.ILogOutput;
import com.android.ddmlib.Log.LogLevel;
import com.google.common.base.Predicate;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import org.echocat.jomon.maven.plugins.ProjectEnabledMojoSupport;
import org.echocat.jomon.process.local.LocalProcessRepository;
import org.echocat.jomon.process.local.daemon.LocalProcessDaemonRepository;
import org.echocat.jomon.process.local.execution.LocalProcessExecuter;
import org.echocat.jomon.runtime.util.Duration;
import org.echocat.jomon.runtime.util.IdEnabled;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.echocat.jomon.process.ExecutableDiscovery.Task.executable;
import static org.echocat.jomon.process.ExecutableDiscovery.executableDiscovery;
import static org.echocat.jomon.runtime.CollectionUtils.asImmutableMap;
import static org.echocat.jomon.runtime.logging.Slf4jUtils.log;
import static org.echocat.jomon.runtime.util.ResourceUtils.closeQuietly;

@SuppressWarnings("InstanceVariableNamingConvention")
public abstract class AndroidMojoSupport extends ProjectEnabledMojoSupport {

    @Nonnull
    protected static final Predicate<IDevice> DEVICE_ONLINE = new Predicate<IDevice>() { @Override public boolean apply(@Nullable IDevice input) {
        return input != null && input.isOnline();
    }};
    @Nonnull
    protected static final AndroidLogHandler ANDROID_LOG_HANDLER = new AndroidLogHandler();

    static {
        Log.setLogOutput(ANDROID_LOG_HANDLER);
    }

    @Nullable
    private LocalProcessDaemonRepository _processDaemonRepository;
    @Nullable
    private LocalProcessRepository _processRepository;

    @Parameter(property = "adb.executable")
    private String adbExecutable;

    @Nullable
    protected String getAdbExecutable() throws MojoExecutionException {
        return adbExecutable;
    }

    @Nonnull
    protected File getTargetAdbExecutable() throws MojoExecutionException {
        final String executableFileName = getAdbExecutable();
        final File executable = executableDiscovery().discover(executable(executableFileName != null ? executableFileName : "adb")
            .withinEnvironmentVariables("ANDROID_HOME")
            .searchInSubDirectories("platform-tools")
        );
        if (executable == null) {
            throw new IllegalArgumentException("Could not find adb executable '" + executableFileName + "' neither as absolute path nor in environment variable ANDROID_HOME nor in system PATH.");
        }
        return executable;
    }

    @Nonnull
    protected LocalProcessDaemonRepository processDaemonRepository() {
        synchronized (this) {
            if (_processDaemonRepository == null) {
                _processDaemonRepository = LocalProcessDaemonRepository.processDaemonRepository();
            }
            return _processDaemonRepository;
        }
    }

    @Nonnull
    protected LocalProcessRepository processRepository() {
        synchronized (this) {
            if (_processRepository == null) {
                _processRepository = LocalProcessRepository.processRepository();
            }
            return _processRepository;
        }
    }

    @Nonnull
    protected LocalProcessExecuter processExecuter() {
        return new LocalProcessExecuter(processRepository());
    }

    @Nullable
    protected Long findPidOf(@Nonnull String pidProperty) throws MojoExecutionException {
        final String plainPid = getProjectProperty(pidProperty);
        Long pid;
        if (!isEmpty(plainPid)) {
            try {
                pid = Long.parseLong(plainPid);
            } catch (final NumberFormatException ignored) {
                pid = null;
            }
        } else {
            pid = null;
        }
        return pid;
    }

    protected <P extends AutoCloseable & IdEnabled<?>> void registerPid(@Nonnull P process, @Nonnull String pidProperty) throws MojoExecutionException {
        validatePidPropertyForRegister(process, pidProperty);
        setProjectProperty(pidProperty, process.getId().toString());
    }

    protected  <P extends AutoCloseable & IdEnabled<?>> void validatePidPropertyForRegister(@Nonnull P process, @Nonnull String pidProperty) throws MojoExecutionException {
        final String pidValue = findProjectProperty(pidProperty);
        if (!isEmpty(pidValue)) {
            closeQuietly(process);
            throw new MojoExecutionException("The project property '" + pidProperty + "' is already set to '" + pidValue + "' this indicates double usage of this maven plugin without configuring another pid property.");
        }
    }

    protected void unregisterPid(@Nonnull String pidProperty) throws MojoExecutionException {
        removeProjectProperty(pidProperty);
    }

    @Nonnull
    protected Duration getDuration(@Nullable String plainDuration, @Nonnull String propertyName) throws MojoExecutionException {
        try {
            return new Duration(get(plainDuration, propertyName));
        } catch (final IllegalArgumentException e) {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }

    protected static class AndroidLogHandler implements ILogOutput {

        @Nonnull
        private static final Map<LogLevel, org.echocat.jomon.runtime.logging.LogLevel> ANDROID_TO_SLF4J_LEVEL = asImmutableMap(
            LogLevel.DEBUG, org.echocat.jomon.runtime.logging.LogLevel.debug,
            LogLevel.INFO, org.echocat.jomon.runtime.logging.LogLevel.info,
            LogLevel.WARN, org.echocat.jomon.runtime.logging.LogLevel.warning,
            LogLevel.ERROR, org.echocat.jomon.runtime.logging.LogLevel.error,
            LogLevel.ASSERT, org.echocat.jomon.runtime.logging.LogLevel.error
        );

        @Override
        public void printLog(LogLevel logLevel, String tag, String message) {
            log(LoggerFactory.getLogger(tag), ANDROID_TO_SLF4J_LEVEL.get(logLevel), message);
        }

        @Override
        public void printAndPromptLog(LogLevel logLevel, String tag, String message) {
            printLog(logLevel, tag, message);
        }

    }
}
