package org.echocat.rundroid.maven.plugins.emulator;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.echocat.jomon.process.CouldNotStartException;
import org.echocat.jomon.process.local.LocalGeneratedProcess;
import org.echocat.jomon.process.local.LocalGeneratedProcessRequirement;
import org.echocat.jomon.process.local.LocalProcessRepository;
import org.echocat.jomon.process.local.daemon.LocalProcessDaemon;
import org.echocat.jomon.runtime.generation.Generator;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import java.io.IOException;
import java.net.BindException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import static java.net.InetAddress.getLocalHost;
import static org.echocat.jomon.process.local.LocalGeneratedProcessRequirement.process;


@ThreadSafe
public class EmulatorDaemon extends LocalProcessDaemon<EmulatorDaemonRequirement> {

    @Nonnull
    private static final Map<LocalGeneratedProcess, String> PROCESS_TO_SERIAL_NUMBER = new WeakHashMap<>();

    @Nullable
    public static String findSerialNumberFor(@Nonnull LocalGeneratedProcess process) {
        synchronized (PROCESS_TO_SERIAL_NUMBER) {
            return PROCESS_TO_SERIAL_NUMBER.get(process);
        }
    }

    @Nonnull
    private final EmulatorDaemonRequirement _requirement;

    @Nullable
    private Integer _port;

    public EmulatorDaemon(@Nonnull LocalProcessRepository processRepository, @Nonnull EmulatorDaemonRequirement requirement) throws CouldNotStartException {
        super(processRepository, requirement);
        _requirement = requirement;
    }

    @Nonnull
    @Override
    protected LocalGeneratedProcess generateProcess(@Nonnull Generator<LocalGeneratedProcess, LocalGeneratedProcessRequirement> repository, @Nonnull EmulatorDaemonRequirement requirement) throws Exception {
        final List<String> arguments = requirement.getArguments();
        if (arguments.contains("-port")) {
            throw new MojoFailureException("Fix setting of a port for the emulator is currently not supported! It is required that the detector is able to find the free port automatically.");
        }
        final int port = findFreePort();
        final LocalGeneratedProcess process = repository.generate(process(requirement.getExecutable())
            .whichIsDaemon(requirement.isShutdownWithThisJvm())
            .withArguments("-avd", requirement.getAvd(), "-port", Integer.toString(port))
            .withArguments(arguments)
        );
        _port = port;
        synchronized (PROCESS_TO_SERIAL_NUMBER) {
            PROCESS_TO_SERIAL_NUMBER.put(process, "emulator-" + port);
        }
        return process;
    }

    @Nonnegative
    private int findFreePort() throws Exception {
        final InetAddress address = getLocalHost();
        Integer result = null;
        for (int port = 5554; result == null && port <= 5584; port += 2) {
            if (isPortFree(address, port) && isPortFree(address, port + 1)) {
                result = port;
            }
        }
        if (result == null) {
            throw new MojoFailureException("Could not find any free port between 5554 and 5584 to bind the emulator on.");
        }
        return result;
    }

    protected boolean isPortFree(@Nonnull InetAddress address, @Nonnegative int port) throws Exception {
        boolean result;
        try {
            final ServerSocket socket = new ServerSocket();
            socket.bind(new InetSocketAddress(address, port));
            socket.close();
            result = true;
        } catch (final BindException ignored) {
            result = false;
        } catch (final IOException e) {
            throw new MojoExecutionException("Could not find test the port " + port + " on '" + address + "'.", e);
        }
        return result;
    }

    @Nullable
    public int getPort() {
        final Integer port = _port;
        if (port == null) {
            throw new IllegalStateException("The emulator is currently not running.");
        }
        return port;
    }

    @Nonnull
    public String getSerialNumber() {
        return "emulator-" + getPort();
    }

    @Override
    public void close() {
        try {
            _port = null;
        } finally {
            super.close();
        }
    }

    @Nonnull
    public String getAvd() {
        return _requirement.getAvd();
    }

    @Nonnull
    public List<String> getArguments() {
        return _requirement.getArguments();
    }

    @Override
    public String toString() {
        return "Emulator{port=" + _port + ", avd=" + getAvd() + "}";
    }

}
