package org.echocat.rundroid.maven.plugins.emulator;

import org.echocat.jomon.process.listeners.startup.StartupListener;
import org.echocat.jomon.process.listeners.startup.StartupListener.NoopStartupListener;
import org.echocat.jomon.process.local.LocalGeneratedProcess;
import org.echocat.rundroid.maven.plugins.utils.DaemonRequirementSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.File;

import static org.echocat.jomon.process.ProcessUtils.toEscapedCommandLine;
import static org.echocat.rundroid.maven.plugins.emulator.EmulatorDaemon.findSerialNumberFor;

public class EmulatorDaemonRequirement extends DaemonRequirementSupport<EmulatorDaemonRequirement, EmulatorDaemon> {

    private static final Logger LOG = LoggerFactory.getLogger(EmulatorDaemonRequirement.class);

    @Nonnull
    public static EmulatorDaemonRequirement emulatorDaemon(@Nonnull File executable, @Nonnull String avd) {
        return new EmulatorDaemonRequirement(executable, avd);
    }

    @Nonnull
    private final File _executable;
    @Nonnull
    private final String _avd;
    @Nonnull
    private File _adbExecutable;

    public EmulatorDaemonRequirement(@Nonnull File executable, @Nonnull String avd) {
        _executable = executable;
        _avd = avd;
    }

    @Nonnull
    public EmulatorDaemonRequirement withAdbExecutable(@Nonnull File adbExecutable) {
        _adbExecutable = adbExecutable;
        return this;
    }

    @Nonnull
    @Override
    public Class<EmulatorDaemon> getType() {
        return EmulatorDaemon.class;
    }

    @Nonnull
    public File getExecutable() {
        return _executable;
    }

    @Nonnull
    public String getAvd() {
        return _avd;
    }

    @Nonnull
    public File getAdbExecutable() {
        return _adbExecutable;
    }

    @Override
    protected void logStartOfProcess(@Nonnull LocalGeneratedProcess process) {
        LOG.info("Starting emulator: " + toEscapedCommandLine(process.getArguments()) + "...");
    }

    @Override
    protected void logStartupDoneOfProcess(@Nonnull LocalGeneratedProcess process) {
        LOG.info("Emulator #" + process.getId() + " (" + findSerialNumberFor(process) + ") started successful.");
    }

    @Override
    protected void logTerminationOfProcess(@Nonnull LocalGeneratedProcess process, boolean success) {
        if (success) {
            LOG.info("Emulator #" + process.getId() + " (" + findSerialNumberFor(process) + ") exited successful. ExitCode: " + process.exitValue());
        } else {
            LOG.warn("Emulator #" + process.getId() + " (" + findSerialNumberFor(process) + ") exited with error. ExitCode: " + process.exitValue());
        }
    }

    @Nonnull
    @Override
    public StartupListener<LocalGeneratedProcess> getStartupListener() {
        return new NoopStartupListener<>();
    }

}
