/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.logcat;

import com.android.ddmlib.IDevice;
import com.google.common.base.Predicate;
import org.echocat.jomon.process.listeners.stream.StreamListener;
import org.echocat.jomon.runtime.util.Consumer;
import org.echocat.jomon.runtime.util.Duration;
import org.echocat.jomon.runtime.util.IdEnabled;
import org.echocat.rundroid.maven.plugins.platform.AdbProcess;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import static org.echocat.jomon.process.ProcessUtils.toEscapedCommandLine;
import static org.echocat.jomon.process.listeners.stream.StreamListeners.redirectToLogger;
import static org.echocat.jomon.process.listeners.stream.StreamListeners.streamListenerFor;
import static org.echocat.jomon.runtime.CollectionUtils.asImmutableList;
import static org.echocat.rundroid.maven.plugins.platform.AdbController.Environment.adbEnvironment;
import static org.echocat.rundroid.maven.plugins.platform.AdbController.adbController;

public class Logcat implements AutoCloseable, IdEnabled<Long> {

    private static final Logger LOG = LoggerFactory.getLogger(Logcat.class);

    @Nonnull
    private static final AtomicLong IDS = new AtomicLong(1);

    @Nonnegative
    private final long _id = IDS.getAndIncrement();
    @Nonnull
    private final Predicate<IDevice> _devicePredicate;
    @Nonnull
    private final File _adbExecutable;

    @Nonnull
    private StreamListener<AdbProcess> _streamListener = defaultStreamListener();
    @Nullable
    private List<String> _arguments;
    @Nonnull
    private Duration _deviceTimeout = new Duration("2m");
    @Nonnull
    private Duration _adbTimeout = new Duration("30s");

    private volatile boolean _logStart;
    private volatile boolean _logStartupDone;
    private volatile boolean _logTermination;

    @Nullable
    private AdbProcess _process;

    public Logcat(@Nonnull Predicate<IDevice> devicePredicate, @Nonnull File adbExecutable) {
        _devicePredicate = devicePredicate;
        _adbExecutable = adbExecutable;
    }

    public void start() throws Exception {
        synchronized (this) {
            if (_process == null) {
                if (isLogStart()) {
                    LOG.info("Starting " + toEscapedCommandLine("logcat", getArguments()) + "...");
                }
                adbController().doWithDevices(
                    adbEnvironment(getAdbExecutable())
                        .withDeviceTimeout(getDeviceTimeout())
                        .withAdbTimeout(getAdbTimeout())
                        .acceptingBy(getDevicePredicate())
                    , deviceConsumer()
                );
                if (isLogStartupDone()) {
                    LOG.info("Logcat #" + getId() + " started successful.");
                }
            }
        }
    }

    @Nonnull
    protected Consumer<IDevice, Exception> deviceConsumer() {
        return new Consumer<IDevice, Exception>() { @Override public void consume(@Nullable IDevice device) throws Exception {
            _process = new AdbProcess(_id, "logcat", getArguments(), device, null, _streamListener, new Consumer<Exception, RuntimeException>() {
                @Override
                public void consume(@Nullable Exception e) {
                    if (isLogTermination()) {
                        if (e != null) {
                            LOG.error("Logcat #" + getId() + " exited with errors. Got: " + e.getMessage(), e);
                        } else {
                            LOG.info("Logcat #" + getId() + " exited successful.");
                        }
                    }
                }
            });
        }};
    }

    @Nonnull
    public File getAdbExecutable() {
        return _adbExecutable;
    }

    @Nonnull
    public Predicate<IDevice> getDevicePredicate() {
        return _devicePredicate;
    }

    @Override
    @Nonnegative
    public Long getId() {
        return _id;
    }

    @Nullable
    public AdbProcess getProcess() {
        return _process;
    }

    @Override
    public void close() throws Exception {
        synchronized (this) {
            if (_process != null) {
                try {
                    _process.close();
                } finally {
                    _process = null;
                }
            }
        }
    }

    @Nonnull
    public Logcat withArguments(@Nullable Iterable<String> arguments) {
        _arguments = asImmutableList(arguments);
        return thisObject();
    }

    @Nonnull
    public Logcat withArguments(@Nullable String... arguments) {
        return withArguments(asImmutableList(arguments));
    }

    @Nonnull
    public Logcat withDeviceTimeout(@Nonnull Duration duration) {
        _deviceTimeout = duration;
        return thisObject();
    }

    @Nonnull
    public Logcat withDeviceTimeout(@Nonnull String duration) {
        return withDeviceTimeout(new Duration(duration));
    }

    @Nonnull
    public Logcat withAdbTimeout(@Nonnull Duration duration) {
        _adbTimeout = duration;
        return thisObject();
    }

    @Nonnull
    public Logcat withAdbTimeout(@Nonnull String duration) {
        return withAdbTimeout(new Duration(duration));
    }

    @Nonnull
    public Logcat withStreamListener(@Nonnull StreamListener<AdbProcess> streamListener) {
        _streamListener = streamListener;
        return thisObject();
    }

    @Nonnull
    public Logcat withStreamListener(@Nonnull String streamListener) {
        return withStreamListener(streamListenerFor(AdbProcess.class, streamListener, defaultStreamListener()));
    }

    @Nonnull
    public Logcat whichLogsStart(boolean log) {
        _logStart = log;
        return thisObject();
    }

    @Nonnull
    public Logcat whichLogsStartupDone(boolean log) {
        _logStartupDone = log;
        return thisObject();
    }

    @Nonnull
    public Logcat whichLogsTermination(boolean log) {
        _logTermination = log;
        return thisObject();
    }

    @Nullable
    public List<String> getArguments() {
        return asImmutableList(_arguments);
    }

    @Nonnull
    public StreamListener<?> getStreamListener() {
        return _streamListener;
    }

    public boolean isLogStart() {
        return _logStart;
    }

    public boolean isLogStartupDone() {
        return _logStartupDone;
    }

    public boolean isLogTermination() {
        return _logTermination;
    }

    @Nonnull
    public Duration getDeviceTimeout() {
        return _deviceTimeout;
    }

    @Nonnull
    public Duration getAdbTimeout() {
        return _adbTimeout;
    }

    @Nonnull
    protected Logcat thisObject() {
        return this;
    }

    @Nonnull
    protected static StreamListener<AdbProcess> defaultStreamListener() {
        return redirectToLogger(Logcat.class);
    }
}
