/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.platform;

import com.android.ddmlib.AndroidDebugBridge;
import com.android.ddmlib.IDevice;
import com.google.common.base.Predicate;
import org.echocat.jomon.runtime.concurrent.RetryForSpecifiedTimeStrategy;
import org.echocat.jomon.runtime.concurrent.RetryingStrategy;
import org.echocat.jomon.runtime.concurrent.StopWatch;
import org.echocat.jomon.runtime.numbers.IntegerRange;
import org.echocat.jomon.runtime.util.Consumer;
import org.echocat.jomon.runtime.util.Duration;
import org.echocat.rundroid.maven.plugins.utils.DeviceUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.WillNotClose;
import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import static com.android.ddmlib.AndroidDebugBridge.createBridge;
import static com.android.ddmlib.AndroidDebugBridge.init;
import static com.google.common.collect.Collections2.filter;
import static java.lang.System.getProperty;
import static java.lang.System.out;
import static java.util.concurrent.Executors.newCachedThreadPool;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.echocat.jomon.runtime.CollectionUtils.asList;
import static org.echocat.jomon.runtime.concurrent.Retryer.executeWithRetry;

@SuppressWarnings({"DuplicateThrows", "UseOfSystemOutOrSystemErr"})
public class AdbController {

    @Nonnull
    private static final Logger LOG = LoggerFactory.getLogger(AdbController.class);
    @Nonnull
    private static final AdbController INSTANCE = new AdbController();

    @Nonnull
    public static final Duration DISPLAY_COUNTDOWN_AFTER = new Duration(getProperty(AdbController.class.getName() + ".displayCountdownAfter", "3s"));

    static {
        init(false);
    }

    @Nonnull
    public static AdbController getInstance() {
        return INSTANCE;
    }

    @Nonnull
    public static AdbController adbController() {
        return getInstance();
    }

    public void doWithDevices(@Nonnull Environment environment, @Nonnull final Consumer<IDevice, Exception> deviceConsumer) throws InterruptedException, Exception {
        final StopWatch stopWatch = new StopWatch();
        final Collection<IDevice> devices = getDevicesFor(environment, stopWatch);
        final ExecutorService threadPool = newCachedThreadPool();
        final List<Future<Void>> futures = new ArrayList<>();
        for (final IDevice device : devices) {
            if (environment.matchingDevice(device)) {
                futures.add(threadPool.submit(new Callable<Void>() { @Override public Void call() throws Exception {
                    deviceConsumer.consume(device);
                    return null;
                }}));
            }
        }
        waitFor(futures);
    }

    protected void waitFor(@Nonnull Iterable<Future<Void>> futures) throws InterruptedException, Exception {
        for (final Future<Void> future : futures) {
            try {
                future.get();
            } catch (final ExecutionException e) {
                final Throwable cause = e.getCause();
                if (cause instanceof Error) {
                    //noinspection ThrowInsideCatchBlockWhichIgnoresCaughtException
                    throw (Error) cause;
                } else if (cause instanceof Exception) {
                    //noinspection ThrowInsideCatchBlockWhichIgnoresCaughtException
                    throw (Exception) cause;
                } else {
                    throw new RuntimeException("Execution produces an exception.", cause != null ? cause : e);
                }
            }
        }
    }

    @Nonnull
    protected Collection<IDevice> getDevicesFor(@Nonnull Environment environment, @Nonnull StopWatch stopWatch) throws TimeoutException, IOException {
        final AndroidDebugBridge adb = getAdbFor(environment);
        return getDevicesFor(environment, adb, leftDeviceTimeoutFor(environment, stopWatch));
    }

    @Nonnull
    protected Collection<IDevice> getDevicesFor(@Nonnull final Environment environment, @Nonnull final AndroidDebugBridge adb, @Nonnull final Duration timeout) throws TimeoutException, IOException {
        final StopWatch stopWatch = new StopWatch();
        final RetryingStrategy<Boolean> strategy = RetryForSpecifiedTimeStrategy.<Boolean>retryForSpecifiedTimeOf(timeout).withWaitBetweenEachTry("1s").withResultsThatForceRetry(false);
        final AtomicReference<Collection<IDevice>> result = new AtomicReference<>();
        final AtomicInteger cleanupConsoleLength = new AtomicInteger();
        try {
            executeWithRetry(new Callable<Boolean>() { @Override public Boolean call() throws Exception {
                displayDeviceCountdownIfNeeded(environment, stopWatch, cleanupConsoleLength);
                final Collection<IDevice> devices = getDevicesFor(environment, adb);
                result.set(devices);
                return environment.matchingNumberOfDevices(devices.size());
            }}, strategy);
            if (!environment.matchingNumberOfDevices(result.get().size())) {
                throw new TimeoutException("Could not get device list within " + timeout + ".");
            }
        } finally {
            cleanupConsoleIfNeeded(environment, cleanupConsoleLength);
        }
        final Collection<IDevice> devices = result.get();
        logFoundDevices(devices);
        return devices;
    }

    protected void logFoundDevices(@Nonnull Collection<IDevice> devices) {
        if (devices.isEmpty()) {
            LOG.info("Found no device.");
        } else if (devices.size() == 1) {
            LOG.info("Using device: " + DeviceUtils.toString(devices.iterator().next()));
        } else {
            final StringBuilder sb = new StringBuilder();
            for (final IDevice device : devices) {
                if (sb.length() > 0) {
                    sb.append(", ");
                }
                sb.append(DeviceUtils.toString(device));
            }
            LOG.info("Using devices: " + sb.toString());
        }
    }

    @Nonnull
    protected Collection<IDevice> getDevicesFor(@Nonnull Environment environment, @Nonnull AndroidDebugBridge adb) {
        final Collection<IDevice> devices = asList(adb.getDevices());
        final Predicate<IDevice> predicate = environment.getDevicePredicate();
        return predicate != null ? filter(devices, predicate) : devices;
    }

    @Nonnull
    protected AndroidDebugBridge getAdbFor(@Nonnull Environment environment) throws TimeoutException, IOException {
        final StopWatch stopWatch = new StopWatch();
        final AndroidDebugBridge adb = createBridge(environment.getExecutable().getPath(), false);
        waitUntilConnected(adb, environment, stopWatch);
        waitForInitialDeviceList(adb, environment, stopWatch);
        return adb;
    }

    protected void waitUntilConnected(@Nonnull final AndroidDebugBridge adb, @Nonnull final Environment environment, @Nonnull final StopWatch stopWatch) throws TimeoutException, IOException {
        final Duration timeout = leftAdbTimeoutFor(environment, stopWatch);
        final RetryingStrategy<Boolean> strategy = RetryForSpecifiedTimeStrategy.<Boolean>retryForSpecifiedTimeOf(timeout).withWaitBetweenEachTry("1s").withResultsThatForceRetry(false);
        final AtomicInteger cleanupConsoleLength = new AtomicInteger();
        try {
            executeWithRetry(new Callable<Boolean>() { @Override public Boolean call() throws Exception {
                displayAdbCountdownIfNeeded(environment, stopWatch, cleanupConsoleLength);
                return adb.isConnected();
            }}, strategy);
            if (!adb.isConnected()) {
                throw new TimeoutException("Could not connect to ADB within " + timeout + ".");
            }
        } finally {
            cleanupConsoleIfNeeded(environment, cleanupConsoleLength);
        }
    }

    protected void waitForInitialDeviceList(@Nonnull final AndroidDebugBridge adb, @Nonnull final Environment environment, @Nonnull final StopWatch stopWatch) throws TimeoutException, IOException {
        final Duration timeout = leftAdbTimeoutFor(environment, stopWatch);
        final RetryingStrategy<Boolean> strategy = RetryForSpecifiedTimeStrategy.<Boolean>retryForSpecifiedTimeOf(timeout).withWaitBetweenEachTry("1s").withResultsThatForceRetry(false);
        final AtomicInteger cleanupConsoleLength = new AtomicInteger();
        try {
            executeWithRetry(new Callable<Boolean>() { @Override public Boolean call() throws Exception {
                displayAdbCountdownIfNeeded(environment, stopWatch, cleanupConsoleLength);
                return adb.hasInitialDeviceList();
            }}, strategy);
            if (!adb.hasInitialDeviceList()) {
                throw new TimeoutException("Did not receive initial device list from ADB after " + timeout + ".");
            }
        } finally {
            cleanupConsoleIfNeeded(environment, cleanupConsoleLength);
        }
    }

    protected void displayDeviceCountdownIfNeeded(@Nonnull Environment environment, @Nonnull StopWatch stopWatch, @Nonnull AtomicInteger cleanupConsoleLength) throws IOException {
        if (shouldDisplayCountdownFor(stopWatch)) {
            final Writer consumer = environment.getProgressConsumer();
            if (consumer != null) {
                final String output = "\rWaiting for devices... " + leftDeviceTimeoutFor(environment, stopWatch).trim(SECONDS);
                consumer.write(output);
                consumer.flush();
                cleanupConsoleLength.set(output.length());
            }
        }
    }

    protected void displayAdbCountdownIfNeeded(@Nonnull Environment environment, @Nonnull StopWatch stopWatch, @Nonnull AtomicInteger cleanupConsoleLength) throws IOException {
        if (shouldDisplayCountdownFor(stopWatch)) {
            final Writer consumer = environment.getProgressConsumer();
            if (consumer != null) {
                final String output = "\rWaiting for adb... " + leftAdbTimeoutFor(environment, stopWatch).trim(SECONDS);
                consumer.write(output);
                consumer.flush();
                cleanupConsoleLength.set(output.length());
            }
        }
    }

    protected boolean shouldDisplayCountdownFor(@Nonnull StopWatch stopWatch) {
        return DISPLAY_COUNTDOWN_AFTER.isLessThanOrEqualTo(stopWatch.getCurrentDuration());
    }

    protected void cleanupConsoleIfNeeded(@Nonnull Environment environment, @Nonnull AtomicInteger cleanupConsoleLength) throws IOException {
        cleanupConsoleIfNeeded(environment, cleanupConsoleLength.get());
    }

    protected void cleanupConsoleIfNeeded(@Nonnull Environment environment, @Nonnull int cleanupConsoleLength) throws IOException {
        if (cleanupConsoleLength > 0) {
            final Writer consumer = environment.getProgressConsumer();
            consumer.write('\r');
            for (int i = 0; i < cleanupConsoleLength; i++) {
                consumer.write(' ');
            }
            consumer.write('\r');
            consumer.flush();
        }
    }

    @Nonnull
    protected Duration leftAdbTimeoutFor(@Nonnull Environment environment, @Nonnull StopWatch stopWatch) {
        return environment.getAdbTimeout().minus(stopWatch.getCurrentDuration());
    }

    @Nonnull
    protected Duration leftDeviceTimeoutFor(@Nonnull Environment environment, @Nonnull StopWatch stopWatch) {
        return environment.getDeviceTimeout().minus(stopWatch.getCurrentDuration());
    }

    public static class Environment {

        @Nonnull
        public static Environment adbEnvironment(@Nonnull File executable) {
            return new Environment(executable);
        }

        @Nonnull
        private final File _executable;

        @Nullable
        private Writer _progressConsumer = new OutputStreamWriter(out);
        @Nullable
        private Predicate<IDevice> _devicePredicate;
        @Nullable
        private Predicate<Integer> _expectedNumberOfDevices = new IntegerRange(1, null);
        @Nonnull
        private Duration _adbTimeout = new Duration("30s");
        @Nonnull
        private Duration _deviceTimeout = new Duration("60s");

        public Environment(@Nonnull File executable) {
            _executable = executable;
        }

        @Nonnull
        public Environment acceptingBy(@Nonnull Predicate<IDevice> predicate) {
            _devicePredicate = predicate;
            return this;
        }

        @Nonnull
        public Environment expectedNumberOfDevices(@Nonnull Predicate<Integer> predicate) {
            _expectedNumberOfDevices = predicate;
            return this;
        }

        @Nonnull
        public Environment expectingMinimumNumberOfDevicesIs(int numberOfExpectedDevices) {
            return expectedNumberOfDevices(new IntegerRange(numberOfExpectedDevices, null));
        }

        @Nonnull
        public Environment expectingMinimumOneDevice() {
            return expectingMinimumNumberOfDevicesIs(1);
        }

        @Nonnull
        public Environment withAdbTimeout(@Nonnull Duration timeout) {
            _adbTimeout = timeout;
            return this;
        }

        @Nonnull
        public Environment withAdbTimeout(@Nonnull String timeout) {
            return withAdbTimeout(new Duration(timeout));
        }

        @Nonnull
        public Environment withDeviceTimeout(@Nonnull Duration timeout) {
            _deviceTimeout = timeout;
            return this;
        }

        @Nonnull
        public Environment withDeviceTimeout(@Nonnull String timeout) {
            return withDeviceTimeout(new Duration(timeout));
        }

        @Nonnull
        public Environment withProgressConsumer(@Nullable @WillNotClose OutputStream os) {
            return withProgressConsumer(os != null ? new OutputStreamWriter(os) : null);
        }

        @Nonnull
        public Environment withProgressConsumer(@Nullable @WillNotClose Writer writer) {
            _progressConsumer = writer;
            return this;
        }

        @Nonnull
        public File getExecutable() {
            return _executable;
        }

        @Nullable
        public Predicate<IDevice> getDevicePredicate() {
            return _devicePredicate;
        }

        @Nullable
        public Predicate<Integer> getExpectedNumberOfDevices() {
            return _expectedNumberOfDevices;
        }

        @Nonnull
        public Duration getAdbTimeout() {
            return _adbTimeout;
        }

        @Nonnull
        public Duration getDeviceTimeout() {
            return _deviceTimeout;
        }

        @Nullable
        @WillNotClose
        public Writer getProgressConsumer() {
            return _progressConsumer;
        }

        protected boolean matchingNumberOfDevices(int numberOfDevices) {
            final Predicate<Integer> predicate = _expectedNumberOfDevices;
            return predicate == null || predicate.apply(numberOfDevices);
        }

        protected boolean matchingDevice(@Nullable IDevice device) {
            final boolean result;
            if (device == null) {
                result = false;
            } else {
                final Predicate<IDevice> predicate = _devicePredicate;
                result = predicate == null || predicate.apply(device);
            }
            return result;
        }
    }

    public static class AdbException extends RuntimeException {

        public AdbException(String message) {
            super(message);
        }

        public AdbException(String message, Throwable cause) {
            super(message, cause);
        }
    }

}
