/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.platform;

import com.android.ddmlib.testrunner.TestIdentifier;
import com.android.ddmlib.testrunner.TestResult;
import com.android.ddmlib.testrunner.TestResult.TestStatus;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;

import static com.android.ddmlib.testrunner.TestResult.TestStatus.INCOMPLETE;
import static java.util.Collections.synchronizedMap;
import static org.echocat.jomon.runtime.CollectionUtils.asImmutableMap;
import static org.echocat.jomon.runtime.CollectionUtils.asImmutableSet;

public class TestRunResult {

    @Nonnull
    private final String _testRunName;
    @Nonnull
    private final Map<TestIdentifier, TestResult> _testResults = synchronizedMap(new LinkedHashMap<TestIdentifier, TestResult>());
    @Nonnull
    private final Map<String, String> _runMetrics = new HashMap<>();

    private boolean _isRunComplete;
    private long _elapsedTime;
    private int _numberOfFailedTests;
    private int _numberOfErrorTests;
    private int _numberOfPassedTests;
    private int _numberOfInCompleteTests;
    private String _runFailureError;

    public TestRunResult(@Nonnull String runName) {
        _testRunName = runName;
    }

    public TestRunResult() {
        this("not started");
    }

    @Nonnull
    public String getName() {
        return _testRunName;
    }

    @Nonnull
    public Map<TestIdentifier, TestResult> getTestResults() {
        return asImmutableMap(_testResults);
    }

    public void addMetrics(@Nullable Map<String, String> runMetrics, boolean aggregateMetrics) {
        if (runMetrics != null) {
            if (aggregateMetrics) {
                for (final Map.Entry<String, String> entry : runMetrics.entrySet()) {
                    final String existingValue = _runMetrics.get(entry.getKey());
                    final String combinedValue = combineValues(existingValue, entry.getValue());
                    _runMetrics.put(entry.getKey(), combinedValue);
                }
            } else {
                _runMetrics.putAll(runMetrics);
            }
        }
    }

    @Nullable
    private String combineValues(@Nullable String existingValue, @Nullable String newValue) {
        String result;
        if (existingValue != null) {
            try {
                final Long existingLong = Long.parseLong(existingValue);
                final Long newLong = Long.parseLong(newValue);
                result = Long.toString(existingLong + newLong);
            } catch (final NumberFormatException ignored) {
                // This is not a long... try double...
                try {
                    final Double existingDouble = Double.parseDouble(existingValue);
                    final Double newDouble = Double.parseDouble(newValue);
                    result = Double.toString(existingDouble + newDouble);
                } catch (final NumberFormatException ignored2) {
                    // This is neither a long nor a double... use given newValue...
                    result = newValue;
                }
            }
        } else {
            result = newValue;
        }
        return result;
    }

    @Nonnull
    public Map<String, String> getRunMetrics() {
        return asImmutableMap(_runMetrics);
    }

    @Nonnull
    public Set<TestIdentifier> getCompletedTests() {
        final Set<TestIdentifier> completedTests = new LinkedHashSet<>();
        for (final Map.Entry<TestIdentifier, TestResult> testEntry : getTestResults().entrySet()) {
            if (!testEntry.getValue().getStatus().equals(INCOMPLETE)) {
                completedTests.add(testEntry.getKey());
            }
        }
        return asImmutableSet(completedTests);
    }

    public boolean isRunFailure() {
        return _runFailureError != null;
    }

    public boolean isRunComplete() {
        return _isRunComplete;
    }

    void setRunComplete(boolean runComplete) {
        _isRunComplete = runComplete;
    }

    void addElapsedTime(long elapsedTime) {
        _elapsedTime += elapsedTime;
    }

    void setRunFailureError(@Nullable String errorMessage) {
        _runFailureError = errorMessage;
    }

    @Nonnegative
    public int getNumberOfPassedTests() {
        return _numberOfPassedTests;
    }

    @Nonnegative
    public int getNumTests() {
        return _testResults.size();
    }

    @Nonnegative
    public int getNumCompleteTests() {
        return getNumTests() - getNumIncompleteTests();
    }

    @Nonnegative
    public int getNumberOfFailedTests() {
        return _numberOfFailedTests;
    }

    @Nonnegative
    public int getNumberOfErrorTests() {
        return _numberOfErrorTests;
    }

    @Nonnegative
    public int getNumIncompleteTests() {
        return _numberOfInCompleteTests;
    }

    public boolean hasFailedTests() {
        return getNumberOfErrorTests() > 0 || getNumberOfFailedTests() > 0;
    }

    @Nonnegative
    public long getElapsedTime() {
        return _elapsedTime;
    }

    @Nullable
    public String getRunFailureMessage() {
        return _runFailureError;
    }

    void reportTestStarted(@Nonnull TestIdentifier test) {
        final TestResult result = _testResults.get(test);
        if (result != null) {
            final TestStatus i = result.getStatus();
            if (i == TestStatus.ERROR) {
                _numberOfErrorTests--;
            } else if (i == TestStatus.FAILURE) {
                _numberOfFailedTests--;
            } else if (i == TestStatus.PASSED) {
                _numberOfPassedTests--;
            }
        } else {
            _numberOfInCompleteTests++;
        }
        _testResults.put(test, new TestResult());
    }

    void reportTestFailure(@Nonnull TestIdentifier test, @Nonnull TestStatus status, @Nullable String trace) {
        TestResult result = _testResults.get(test);
        if (result == null) {
            result = new TestResult();
            _testResults.put(test, result);
        } else if (result.getStatus().equals(TestStatus.PASSED)) {
            _numberOfPassedTests--;
        }
        result.setStackTrace(trace);
        if (status == TestStatus.ERROR) {
            _numberOfErrorTests++;
            result.setStatus(TestStatus.ERROR);
        } else if (status == TestStatus.FAILURE) {
            result.setStatus(TestStatus.FAILURE);
            _numberOfFailedTests++;
        }
    }

    boolean reportTestEnded(@Nonnull TestIdentifier test, @Nullable Map<String, String> testMetrics) {
        TestResult result = _testResults.get(test);
        if (result == null) {
            result = new TestResult();
            _testResults.put(test, result);
        } else {
            _numberOfInCompleteTests--;
        }
        result.setEndTime(System.currentTimeMillis());
        result.setMetrics(testMetrics);
        final boolean success;
        if (result.getStatus().equals(INCOMPLETE)) {
            result.setStatus(TestStatus.PASSED);
            _numberOfPassedTests++;
            success = true;
        } else {
            success = false;
        }
        return success;
    }
}
