/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.utils;

import com.google.common.base.Predicate;
import org.echocat.jomon.process.listeners.stream.StreamListener;
import org.echocat.jomon.process.local.LocalGeneratedProcess;
import org.echocat.jomon.process.local.daemon.LocalProcessDaemon;
import org.echocat.jomon.process.local.daemon.LocalProcessDaemonRequirement;
import org.echocat.jomon.runtime.io.StreamType;
import org.echocat.jomon.runtime.numbers.IntegerRange;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;

import static java.util.Collections.unmodifiableList;
import static org.echocat.jomon.process.listeners.stream.StreamListeners.redirectToLogger;
import static org.echocat.jomon.process.listeners.stream.StreamListeners.streamListenerFor;
import static org.echocat.jomon.runtime.CollectionUtils.addAll;

public abstract class DaemonRequirementSupport<T extends DaemonRequirementSupport<T, D>, D extends LocalProcessDaemon<?>> implements LocalProcessDaemonRequirement<D> {

    @Nonnull
    protected static final StreamListener<LocalGeneratedProcess> DEFAULT_STREAM_LISTENER = redirectToLogger();
    @Nonnull
    protected static final IntegerRange EXIT_CODE_VALIDATOR = new IntegerRange(0, 2);

    @Nonnull
    private final StreamListenerDelegate _streamListenerDelegate = new StreamListenerDelegate();
    @Nonnull
    private final List<String> _arguments = new ArrayList<>();

    @Nonnull
    private StreamListener<LocalGeneratedProcess> _streamListener = DEFAULT_STREAM_LISTENER;
    
    private boolean _logStartOfProcess;
    private boolean _logStartupDoneOfProcess;
    private boolean _logTerminationOfProcess;
    private boolean _shutdownWithThisJvm;

    @Nonnull
    public T withStreamListener(@Nonnull StreamListener<LocalGeneratedProcess> streamListener) {
        _streamListener = streamListener;
        return thisObject();
    }

    @Nonnull
    public T withStreamListener(@Nonnull String streamListenerConfiguration) {
        return withStreamListener(streamListenerFor(LocalGeneratedProcess.class, streamListenerConfiguration, DEFAULT_STREAM_LISTENER));
    }

    @Nonnull
    public T withArguments(@Nullable Iterable<String> arguments) {
        addAll(_arguments, arguments);
        return thisObject();
    }

    @Nonnull
    public T withArguments(@Nullable String... arguments) {
        addAll(_arguments, arguments);
        return thisObject();
    }

    @Nonnull
    public T withArgument(@Nonnull String argument) {
        return withArguments(argument);
    }

    @Nonnull
    public List<String> getArguments() {
        return unmodifiableList(_arguments);
    }

    @Nonnull
    public T whichLogsStartOfProcess(boolean log) {
        _logStartOfProcess = log;
        return thisObject();
    }

    @Nonnull
    public T whichLogsStartOfProcess() {
        return whichLogsStartOfProcess(true);
    }

    @Nonnull
    public T whichNotLogStartOfProcess() {
        return whichLogsStartOfProcess(false);
    }

    @Nonnull
    public T whichLogsStartupDoneOfProcess(boolean log) {
        _logStartupDoneOfProcess = log;
        return thisObject();
    }

    @Nonnull
    public T whichLogsStartupDoneOfProcess() {
        return whichLogsStartupDoneOfProcess(true);
    }

    @Nonnull
    public T whichNotLogStartupDoneOfProcess() {
        return whichLogsStartupDoneOfProcess(false);
    }

    @Nonnull
    public T whichLogsTerminationOfProcess(boolean log) {
        _logTerminationOfProcess = log;
        return thisObject();
    }

    @Nonnull
    public T whichLogsTerminationOfProcess() {
        return whichLogsTerminationOfProcess(true);
    }

    @Nonnull
    public T whichNotLogTerminationOfProcess() {
        return whichLogsTerminationOfProcess(false);
    }

    @Nonnull
    public T whichShuttingDownWithThisJvm(boolean shutdownWithThisJvm) {
        _shutdownWithThisJvm = shutdownWithThisJvm;
        return thisObject();
    }

    @Nonnull
    public T whichShuttingDownWithThisJvm() {
        return whichShuttingDownWithThisJvm(true);
    }

    @Nonnull
    public T whichNotShuttingDownWithThisJvm() {
        return whichShuttingDownWithThisJvm(false);
    }

    public boolean isShutdownWithThisJvm() {
        return _shutdownWithThisJvm;
    }

    protected abstract void logStartOfProcess(@Nonnull LocalGeneratedProcess process);

    protected abstract void logStartupDoneOfProcess(@Nonnull LocalGeneratedProcess process);

    protected abstract void logTerminationOfProcess(@Nonnull LocalGeneratedProcess process, boolean success);

    @Nonnull
    @Override
    public StreamListener<LocalGeneratedProcess> getStreamListener() {
        return _streamListenerDelegate;
    }

    @Nonnull
    @Override
    public Predicate<Integer> getExitCodeValidator() {
        return EXIT_CODE_VALIDATOR;
    }

    @Nonnull
    protected T thisObject() {
        // noinspection unchecked
        return (T) this;
    }

    public class StreamListenerDelegate implements StreamListener<LocalGeneratedProcess> {

        @Override
        public boolean canHandleReferenceType(@Nonnull Class<?> type) {
            return _streamListener.canHandleReferenceType(type);
        }

        @Override
        public void notifyProcessStarted(@Nonnull LocalGeneratedProcess process) {
            if (_logStartOfProcess) {
                logStartOfProcess(process);
            }
            _streamListener.notifyProcessStarted(process);
        }

        @Override
        public void notifyProcessStartupDone(@Nonnull LocalGeneratedProcess process) {
            if (_logStartupDoneOfProcess) {
                logStartupDoneOfProcess(process);
            }
            _streamListener.notifyProcessStartupDone(process);
        }

        @Override
        public void notifyOutput(@Nonnull LocalGeneratedProcess process, @Nonnull byte[] data, @Nonnegative int offset, @Nonnegative int length, @Nonnull StreamType streamType) {
            _streamListener.notifyOutput(process, data, offset, length, streamType);
        }

        @Override
        public void flushOutput(@Nonnull LocalGeneratedProcess process, @Nonnull StreamType streamType) {
            _streamListener.flushOutput(process, streamType);
        }

        @Override
        public void notifyProcessTerminated(@Nonnull LocalGeneratedProcess process, boolean success) {
            if (_logTerminationOfProcess) {
                logTerminationOfProcess(process, success);
            }
            _streamListener.notifyProcessTerminated(process, success);
        }
    }

}
