/*
 * Copyright (C) 2011 Eiichiro Uchiumi. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.eiichiro.gig.tools.scaffold;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Map;

/**
 * {@code ScaffoldFile} represents a file being generated by 
 * {@link ScaffoldGenerator}.
 * 
 * @author <a href="mailto:eiichiro@eiichiro.org">Eiichiro Uchiumi</a>
 */
public class ScaffoldFile {

	private final String template;
	
	private final File directory;
	
	private final File file;
	
	private final Map<String, String> parameters;
	
	ScaffoldFile(String template, String directory, String file,
			Map<String, String> parameters) throws IOException {
		ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
		BufferedReader reader = new BufferedReader(new InputStreamReader(classLoader.getResourceAsStream(template)));
		StringBuilder builder = new StringBuilder();
		String line = null;
		
		try {
			while ((line = reader.readLine()) != null) {
				builder.append(line + '\n');
			}
			
			this.template = builder.toString();
		} finally {
			reader.close();
		}
		
		this.directory = new File(directory);
		this.file = new File(directory, file);
		this.parameters = parameters;
	}
	
	boolean exists() {
		return file.exists();
	}
	
	boolean delete() {
		return file.delete();
	}
	
	void save() throws IOException {
		if (!directory.exists()) {
			if (!directory.mkdirs()) {
				throw new IOException("Cannot create directory [" + directory + "]");
			}
		}
		
		String content = template;
		
		for (String key : parameters.keySet()) {
			content = content.replace(key, parameters.get(key));
		}
		
		FileWriter writer = null;
		
		try {
			writer = new FileWriter(file);
			writer.write(content);
		} finally {
			if (writer != null) {
				writer.close();
			}
		}
	}
	
	String path() {
		return file.getPath();
	}
	
}
