package org.ektorp.spring;

import org.ektorp.http.*;
import org.ektorp.support.*;
import org.slf4j.*;
import org.springframework.beans.factory.*;
import org.springframework.beans.factory.annotation.*;
/**
 * FactoryBean that produces a HttpClient.
 * Configuration parameters are set through @Value annotations.
 * 
 * The application context must define properties along the line of:
 * <code>
 * <util:properties id="couchdbProperties" location="classpath:/couchdb.properties"/>
 * </code>
 * @author henrik lundgren
 *
 */
public class HttpClientFactoryBean implements FactoryBean<HttpClient> {

	private final static Logger LOG = LoggerFactory.getLogger(HttpClientFactoryBean.class);
	
	public @Value("#{couchdbProperties.host}") String host;
	public @Value("#{couchdbProperties.port}") int port;
	public @Value("#{couchdbProperties.maxConnections}") int maxConnections;
	public @Value("#{couchdbProperties.connectionTimeout}") int connectionTimeout = 1000;
	public @Value("#{couchdbProperties.socketTimeout}") int socketTimeout = 10000;
	public boolean autoUpdateViewOnChange;
	
	public HttpClient getObject() throws Exception {
		LOG.debug("host: {}", host);
		LOG.debug("port: {}", port);
		LOG.debug("maxConnections: {}", maxConnections);
		LOG.debug("connectionTimeout: {}", connectionTimeout);
		LOG.debug("socketTimeout: {}", socketTimeout);
		
		HttpClient client = new StdHttpClient.Builder()
								.host(host)
								.port(port)
								.maxConnections(maxConnections)
								.connectionTimeout(connectionTimeout)
								.socketTimeout(socketTimeout)
								.build();
		testConnect(client);
		configureAutoUpdateViewOnChange();
		return client;
	}
	
	private void configureAutoUpdateViewOnChange() {
		if (autoUpdateViewOnChange && !Boolean.getBoolean(CouchDbRepositorySupport.AUTO_UPDATE_VIEW_ON_CHANGE)) {
			System.setProperty(CouchDbRepositorySupport.AUTO_UPDATE_VIEW_ON_CHANGE, Boolean.TRUE.toString());
		}
	}

	private void testConnect(HttpClient client) {
		try {
			client.head("/");
		} catch (Exception e) {
			throw new BeanCreationException(String.format("Cannot connect to CouchDb@%s:%s", host, port), e);
		}
	}

	public Class<? extends HttpClient> getObjectType() {
		return HttpClient.class;
	}

	public boolean isSingleton() {
		return true;
	}

	public void setAutoUpdateViewOnChange(boolean b) {
		this.autoUpdateViewOnChange = b;
	}
	
}
