package org.ektorp;

import java.io.*;
import java.util.*;

import org.codehaus.jackson.*;
/**
 * Primary interface for working with Objects mapped as documents in CouchDb.
 * 
 * Mapped Objects must have getters and setters for id and revision.
 * 
 * public String getId()
 * public void setId(String s)
 * public String getRevision()
 * public void setRevision(String s)
 * 
 * Reflection is used to access these methods.
 * 
 * @author henrik lundgren
 *
 */
public interface CouchDbConnector {
	/**
	 * 
	 * @param id
	 * @param The document in the form of an Jackson JsonNode.
	 */
	void create(String id, JsonNode json);
	/**
	 * Creates the Object as a document in the database.
	 * If the id is not set it will be generated by the database.
	 * 
	 * The Object's revision field will be updated through the setRevision(String s) method.
	 * @param o
	 */
	void create(Object o);
	/**
	 * Updates the document.
	 * 
	 * The Object's revision field will be updated through the setRevision(String s) method.
	 * @param o
	 */
	void update(Object o);
	/**
	 * Deletes the Object in the database.
	 * @param o
	 * @return the revision of the deleted document
	 */
	String delete(Object o);
	/**
	 * Deletes the document in the database.
	 * @param id
	 * @param revision
	 * @return the revision of the deleted document
	 */
	String delete(String id, String revision);
	/**
	 * 
	 * @param <T>
	 * @param c the target class to map to.
	 * @param id the id of the document in the database.
	 * @return the document mapped as the specified class.
	 */
	<T> T get(Class<T> c, String id);
	/**
	 * Check if the database contains a document.
	 * @param id
	 * @return true if a document with the id exists in the database
	 */
	boolean contains(String id);
	/**
	 * 
	 * @param docId
	 * @return the document as an InputStream, don't forget to close the stream when finished.
	 */
	InputStream getAsStream(String docId);
	/**
	 * 
	 * @param id
	 * @return
	 */
	List<Revision> getRevisions(String id);
	/**
	 * 
	 * @param id
	 * @param attachmentId
	 * @return
	 */
	Attachment getAttachment(String id, String attachmentId);
	/**
	 * Creates both the document and the attachment
	 * @param docId
	 * @param a - the data to be saved as an attachment 
	 * @return revision of the created attachment document
	 */
	String createAttachment(String docId, Attachment a);
	/**
	 * Adds an attachment to the specified document id.
	 * @param docId
	 * @param revision
	 * @param a - the data to be saved as an attachment
	 * @return the new revision of the document
	 */
	String createAttachment(String docId, String revision, Attachment a);
	/**
	 * 
	 * @param docId
	 * @param revision
	 * @param attachmentId
	 * @return the new revision of the document
	 */
	String deleteAttachment(String docId, String revision, String attachmentId);
	/**
	 * @return all document ids in the database including design document ids.
	 */
	List<String> getAllDocIds();
	/**
	 * This method requires the view result values to be document ids.
	 * 
	 * {"_id":"_design/ExampleDoc",
	 *	"views":{
	 *		"all": {"map": "function(doc) { emit(null, doc._id);}"},
	 *		"by_name": {"map": "function(doc) { emit(doc.name, doc._id);}"}
	 *	}
	 * }
	 * 
	 * @param <T>
	 * @param query
	 * @param type the type to map the result to
	 * @return the view result mapped as the specified class.
	 */
	<T> List<T> queryView(ViewQuery query, Class<T> type);
	/**
	 * 
	 * @param query
	 * @return
	 */
	ViewResult queryView(ViewQuery query);
	/**
	 * 
	 * @param view
	 * @param keys
	 * @return the view result as a raw InputStream.
	 */
	InputStream queryForStream(ViewQuery query);
	/**
	 * Creates a database on the configured path if it does not exists.
	 */
	void createDatabaseIfNotExists();
	/**
	 * @return name
	 */
	String getDatabaseName();
	/**
	 * 
	 * @return
	 */
	String path();
	
}
