package org.ektorp.impl;

import java.io.*;
import java.util.*;

import org.codehaus.jackson.*;
import org.codehaus.jackson.map.*;
import org.ektorp.http.*;
import org.ektorp.util.*;
/**
 * Reads view result and extracts documents and mapps them to the specified type.
 * 
 * Threadsafe.
 * 
 * @author henrik lundgren
 *
 * @param <T>
 */
public class EmbeddedDocViewResponseHandler<T> extends StdResponseHandler<List<T>> {

	private static final String ROWS_FIELD_NAME = "rows";
	private static final String VALUE_FIELD_NAME = "value";
	private static final String INCLUDED_DOC_FIELD_NAME = "doc";
	private static final String TOTAL_ROWS_FIELD_NAME = "total_rows";
	
	private final ObjectMapper mapper;
	private final Class<T> type;
	
	public EmbeddedDocViewResponseHandler(Class<T> docType, ObjectMapper om) {
		Assert.notNull(om, "ObjectMapper may not be null");
		Assert.notNull(docType, "docType may not be null");
		mapper = om;
		type = docType;
	}
	
	@Override
	public List<T> success(HttpResponse hr) throws Exception {
		JsonParser jp = mapper.getJsonFactory().createJsonParser(hr.getContent());
		if (jp.nextToken() != JsonToken.START_OBJECT) {
			throw new RuntimeException("Expected data to start with an Object");
		}
		
		Map<String, Integer> fields = readHeaderFields(jp);
		
		List<T> result;
		if (fields.containsKey(TOTAL_ROWS_FIELD_NAME)) {
			result = new ArrayList<T>(fields.get(TOTAL_ROWS_FIELD_NAME));
		} else {
			result = new ArrayList<T>();
		}
		
		while (jp.nextToken() != null) {
			skipToDoc(jp);
			if (jp.getCurrentToken() != JsonToken.START_OBJECT) {
				break;
			}
			result.add(jp.readValueAs(type));
		}
		return result;
	}
	
	private void skipToDoc(JsonParser jp) throws JsonParseException, IOException {
		while (jp.nextToken() != null) {
			JsonToken token = jp.getCurrentToken();
			if (token != JsonToken.FIELD_NAME) {
				continue;
			}
			
			String nextFieldName = jp.getCurrentName();
			token = jp.nextToken();
			if ((nextFieldName.equals(VALUE_FIELD_NAME) || nextFieldName.equals(INCLUDED_DOC_FIELD_NAME))
					&& token == JsonToken.START_OBJECT) {
				break;
			}
		}
	}

	private Map<String, Integer> readHeaderFields(JsonParser jp) throws JsonParseException, IOException {
		Map<String, Integer> map = new HashMap<String, Integer>();
		jp.nextToken();
		String nextFieldName = jp.getCurrentName();
		while(!nextFieldName.equals(ROWS_FIELD_NAME)) {
			jp.nextToken();
			map.put(nextFieldName, Integer.valueOf(jp.getIntValue()));
			jp.nextToken();
			nextFieldName = jp.getCurrentName();
		}
		return map;
	}
}
