package org.ektorp.impl;

import java.io.*;
import java.util.*;

import org.codehaus.jackson.*;
import org.codehaus.jackson.map.*;
import org.ektorp.*;
import org.ektorp.http.*;
/**
 * Handles responses from view queries when the result should be parsed into a list of Java Objects.
 * A instance of this class is only valid to read a response one time.
 * @author remi.vankeisbelck
 * @author henrik lundgren
 *
 * @param <T>
 */
public class QueryViewResponseHandler<T> extends StdResponseHandler<List<T>> {

    private final ObjectMapper objectMapper;
    private final boolean includeDocs;
    private final Class<T> type;
    private final StdCouchDbConnector db;

    private List<T> result;
    /**
     * If the query has the property includeDocs = "true", this factory is to be used.
     * 
     * @param <T>
     * @param objectMapper
     * @param type
     * @param db
     * @return
     */
    public static <T> QueryViewResponseHandler<T> newDocsIncludedHandler(ObjectMapper objectMapper, Class<T> type) {
    	return new QueryViewResponseHandler<T>(objectMapper, true, type, null);
    }
    /**
     * If the query has the property includeDocs = "false" and the view result contains embedded documents or its id in the value field, this factory is to be used.
     * Since embedded docs occupy discspace or loading each document separately from the database might be slow it is recommended to issue queries with
     * the includeDocs parameter set to true and use the factory method above instead. 
     * 
     * @param <T>
     * @param objectMapper
     * @param type
     * @param db
     * @return
     */
    public static <T> QueryViewResponseHandler<T> newLoadDocsByIdHandler(ObjectMapper objectMapper, Class<T> type, StdCouchDbConnector db) {
    	return new QueryViewResponseHandler<T>(objectMapper, false, type, db);
    }
    
    private QueryViewResponseHandler(ObjectMapper objectMapper, boolean includeDocs, Class<T> type, StdCouchDbConnector db) {
        this.objectMapper = objectMapper;
        this.includeDocs = includeDocs;
        this.type = type;
        this.db = db;
    }

    @Override
    public List<T> success(HttpResponse hr) throws Exception {
    	assertInstanceIsFresh();
        // TODO: reading view result should be done through the Jackson streaming API
        JsonNode root = objectMapper.readValue(hr.getContent(), JsonNode.class);
        JsonNode totalRowsNode = root.get("total_rows"); 
        int totalRows = totalRowsNode != null ? totalRowsNode.getIntValue() : 10;
        result = new ArrayList<T>(totalRows);
        if (totalRows > 0) {
            JsonNode rows = root.get("rows");
            for (JsonNode row : rows) {
                readRow(row);
            }
        }
        return result;
    }
    
	private void assertInstanceIsFresh() {
		if (result != null) {
			throw new IllegalStateException("This response handler has already been used once.");
		}
	}
	
	private void readRow(JsonNode row) throws IOException,
			JsonParseException, JsonMappingException {
		if (includeDocs) {
		    readIncludedDoc(row);
		} else {
		    JsonNode jsonValue = row.get("value");
		    if (jsonValue.isTextual()) {
		        readDocFromDb(row);
		    } else {
		        readEmbeddedDoc(jsonValue);
		    }
		}
	}
	private void readDocFromDb(JsonNode row) {
		String id = row.get("value").getTextValue();
		if (id == null || id.length() == 0) {
		    throw new DbAccessException("view result value field did not contain a document id");
		}
		result.add(db.get(type, id));
	}
	private void readEmbeddedDoc(JsonNode jsonValue) throws IOException,
			JsonParseException, JsonMappingException {
		result.add(objectMapper.treeToValue(jsonValue, type));
	}
	private void readIncludedDoc(JsonNode row) throws IOException,
			JsonParseException, JsonMappingException {
		JsonNode doc = row.get("doc");
		readEmbeddedDoc(doc);
	}

    public List<T> getResult() {
        return result;
    }
}
