package org.ektorp.impl;

import static java.lang.String.*;

import java.util.*;

import org.codehaus.jackson.map.*;
import org.codehaus.jackson.type.*;
import org.ektorp.*;
import org.ektorp.http.*;
import org.ektorp.util.*;
import org.slf4j.*;
/**
 * 
 * @author henrik lundgren
 *
 */
public class StdCouchDbInstance implements CouchDbInstance {

	private final static Logger LOG = LoggerFactory.getLogger(StdCouchDbInstance.class);
	private final static TypeReference<List<String>> STRING_LIST_TYPE_DEF = new TypeReference<List<String>>() {};
	
	private final HttpClient client;
	private final RestTemplate restTemplate;
	private final ObjectMapper objectMapper;
	
	public StdCouchDbInstance(HttpClient client) {
		this(client, new ObjectMapper());
	}
	
	public StdCouchDbInstance(HttpClient client, ObjectMapper om) {
		Assert.notNull(client, "HttpClient may not be null");
		Assert.notNull(om, "ObjectMapper may not be null");
		this.client = client;
		this.restTemplate = new RestTemplate(client);
		this.objectMapper = om;
	}
	
	public void createDatabase(String path) {
		createDatabase(DbPath.fromString(path));
	}
	
	public void createDatabase(DbPath db) {
		List<String> all = getAllDatabases();
		if (all.contains(db.getDbName())) {
			throw new DbAccessException(format("A database with path %s already exists", db.getPath()));
		}
		LOG.debug("creating db path: {}", db.getPath());
		restTemplate.put(db.getPath());
	}

	public void deleteDatabase(String path) {
		Assert.notNull(path);
		restTemplate.delete(DbPath.fromString(path).getPath());
	}

	public List<String> getAllDatabases() {
		return restTemplate.get("/_all_dbs", new StdResponseHandler<List<String>>(){
			@Override
			public List<String> success(HttpResponse hr) throws Exception {
				return objectMapper.readValue(hr.getContent(), STRING_LIST_TYPE_DEF);
			}
		});
	}

	public HttpClient getConnection() {
		return client;
	}
}
