package org.ektorp;

import java.io.*;
import java.util.*;

import org.codehaus.jackson.*;
import org.ektorp.http.*;
/**
 * Primary interface for working with Objects mapped as documents in CouchDb.
 * 
 * Mapped Objects must have getters and setters for id and revision.
 * 
 * public String getId()
 * public void setId(String s)
 * public String getRevision()
 * public void setRevision(String s)
 * 
 * Reflection is used to access these methods.
 * 
 * @author henrik lundgren
 *
 */
public interface CouchDbConnector {
	/**
	 * 
	 * @param id
	 * @param json The document in the form of an Jackson JsonNode.
	 */
	void create(String id, JsonNode json);
	/**
	 * Creates the Object as a document in the database.
	 * If the id is not set it will be generated by the database.
	 * 
	 * The Object's revision field will be updated through the setRevision(String s) method.
	 * @param o
	 */
	void create(Object o);
	/**
	 * Updates the document.
	 * 
	 * The Object's revision field will be updated through the setRevision(String s) method.
	 * @param o
	 */
	void update(Object o);
	/**
	 * Deletes the Object in the database.
	 * @param o
	 * @return the revision of the deleted document
	 */
	String delete(Object o);
	/**
	 * Deletes the document in the database.
	 * @param id
	 * @param revision
	 * @return the revision of the deleted document.
	 */
	String delete(String id, String revision);
	/**
	 * 
	 * @param <T>
	 * @param c the target class to map to.
	 * @param id the id of the document in the database.
	 * @return the document mapped as the specified class.
	 * @throws DocumentNotFoundException if the document was not found.
	 */
	<T> T get(Class<T> c, String id);
	/**
	 * 
	 * @param <T>
	 * @param c the target class to map to.
	 * @param id the id of the document in the database.
	 * @param rev of the object.
	 * @return the document mapped as the specified class.
	 * @throws DocumentNotFoundException if the document was not found.
	 */
	<T> T get(Class<T> c, String id, String rev);
	/**
	 * Check if the database contains a document.
	 * @param id
	 * @return true if a document with the id exists in the database
	 */
	boolean contains(String id);
	/**
	 * Please note that the stream has to be closed after usage,
	 * otherwise http connection leaks will occur and the system will eventually hang due
	 * to connection starvation.
	 * 
	 * @param id
	 * @return the document as raw json in an InputStream, don't forget to close the stream when finished.
	 * @throws DocumentNotFoundException if the document was not found.
	 */
	InputStream getAsStream(String id);
	/**
	 * Please note that the stream has to be closed after usage,
	 * otherwise http connection leaks will occur and the system will eventually hang due
	 * to connection starvation.
	 * 
	 * @param id
	 * @param rev
	 * @return the document as raw json in an InputStream, don't forget to close the stream when finished.
	 * @throws DocumentNotFoundException if the document was not found.
	 */
	InputStream getAsStream(String id, String rev);
	/**
	 * 
	 * @param id
	 * @return
	 */
	List<Revision> getRevisions(String id);
	/**
	 * Reads an attachment from the database.
	 * 
	 * Please note that the stream has to be closed after usage,
	 * otherwise http connection leaks will occur and the system will eventually hang due
	 * to connection starvation.
	 *  
	 * @param id
	 * @param attachmentId
	 * @return the attachment in the form of an AttachmentInputStream.
	 * @throws DocumentNotFoundException if the document was not found.
	 */
	AttachmentInputStream getAttachment(String id, String attachmentId);
	/**
	 * Creates both the document and the attachment
	 * @param docId
	 * @param a - the data to be saved as an attachment 
	 * @return revision of the created attachment document
	 */
	String createAttachment(String docId, AttachmentInputStream data);
	/**
	 * Adds an attachment to the specified document id.
	 * @param docId
	 * @param revision
	 * @param a - the data to be saved as an attachment
	 * @return the new revision of the document
	 */
	String createAttachment(String docId, String revision, AttachmentInputStream data);
	/**
	 * 
	 * @param docId
	 * @param revision
	 * @param attachmentId
	 * @return the new revision of the document
	 */
	String deleteAttachment(String docId, String revision, String attachmentId);
	/**
	 * @return all document ids in the database including design document ids.
	 */
	List<String> getAllDocIds();

	/**
	 * This method requires the view result values to be document ids or documents :
     * <ul>
     * <li>If the value is a document id, then the document is fetched from couchDB.</li>
     * <li>If the value is a document, then it is used directly for unmarshalling.</li>
	 * </ul>
     *
	 * {"_id":"_design/ExampleDoc",
	 *	"views":{
	 *		"all": {"map": "function(doc) { emit(null, doc._id);}"},
     *		"by_name": {"map": "function(doc) { emit(doc.name, doc._id);}"} // emit doc id
     *		"by_author": {"map": "function(doc) { emit(doc.author, doc);}"} // emit doc
	 *	}
	 * }
	 * 
	 * @param <T>
	 * @param query
	 * @param type the type to map the result to
	 * @return the view result mapped as the specified class.
	 */
	<T> List<T> queryView(ViewQuery query, Class<T> type);

	/**
	 * 
	 * @param query
	 * @return
	 */
	ViewResult queryView(ViewQuery query);
	/**
	 * 
	 * @param query
	 * @return the view result as a raw InputStream.
	 */
	InputStream queryForStream(ViewQuery query);
	/**
	 * Creates a database on the configured path if it does not exists.
	 */
	void createDatabaseIfNotExists();
	/**
	 * @return name
	 */
	String getDatabaseName();
	/**
	 * 
	 * @return
	 */
	String path();
	/**
	 * Convenience method for accessing the underlying HttpClient.
	 * Preferably used wrapped in a org.ektorp.http.RestTemplate.
	 * @return
	 */
	HttpClient getConnection();
	/**
	 * Provides meta information about this database.
	 * @return
	 */
	DbInfo getDbInfo();
	/**
	 * Compaction compresses the database file by removing unused sections created during updates.
	 * This call is non-blocking, a compaction background task will be created on the CouchDB instance.
	 */
	void compact();
	/**
	 * This compacts the view index from the current version of the design document.
	 * This call is non-blocking, a compaction background task will be created on the CouchDB instance.
	 * @param designDocumentId
	 */
	void compactViews(String designDocumentId);
	/**
	 * View indexes on disk are named after their MD5 hash of the view definition.
	 * When you change a view, old indexes remain on disk.
	 * To clean up all outdated view indexes (files named after the MD5 representation of views, that does not exist anymore) you can trigger a view cleanup
	 */
	void cleanupViews();
	/**
	 * Revision limit defines a upper bound of document revisions which CouchDB keeps track of
	 * @return
	 */
	int getRevisionLimit();
	
	void setRevisionLimit(int limit);
	/**
	 * Replicate the content in the source database into this database.
	 * @param source database
	 * @return ReplicationStatus
	 */
	ReplicationStatus replicateFrom(String source);
	/**
	 * Replicate the content in this database into the specified target database.
	 * The target must exist.
	 * @param target database
	 * @return ReplicationStatus
	 */
	ReplicationStatus replicateTo(String target);
	/**
	 * Creates, updates or deletes all objects in the supplied collection.
	 * Some documents may successfully be saved and some may not.
	 * The response will tell the application which documents were saved or not. In the case of a power failure, when the database restarts some may have been saved and some not.
	 * @param objects, all objects will have their id and revision set.
	 * @return The list will only contain entries for documents that has any kind of error code returned from CouchDB. i.e. the list will be empty if everything was completed successfully.
	 */
	List<DocumentOperationResult> executeBulk(Collection<?> objects);
	/**
	 * Creates, updates or deletes all objects in the supplied collection.
	 * In the case of a power failure, when the database restarts either all the changes will have been saved or none of them.
	 * However, it does not do conflict checking, so the documents will be committed even if this creates conflicts.
	 * @param objects, all objects will have their id and revision set.
	 * @return The list will only contain entries for documents that has any kind of error code returned from CouchDB. i.e. the list will be empty if everything was completed successfully.
	 */
	List<DocumentOperationResult> executeAllOrNothing(Collection<?> objects);
}
