package org.ektorp.impl;

import java.io.*;
import java.util.*;

import org.codehaus.jackson.*;
import org.codehaus.jackson.map.*;
import org.ektorp.*;
import org.ektorp.http.*;
import org.ektorp.util.*;
/**
 * Reads view result and extracts documents and maps them to the specified type.
 * 
 * Threadsafe.
 * 
 * @author henrik lundgren
 *
 * @param <T>
 */
public class EmbeddedDocViewResponseHandler<T> extends StdResponseHandler<List<T>> {

	private static final String ROWS_FIELD_NAME = "rows";
	private static final String VALUE_FIELD_NAME = "value";
	private static final String INCLUDED_DOC_FIELD_NAME = "doc";
	private static final String TOTAL_ROWS_FIELD_NAME = "total_rows";
	
	private final ObjectMapper mapper;
	private final Class<T> type;
	
	public EmbeddedDocViewResponseHandler(Class<T> docType, ObjectMapper om) {
		Assert.notNull(om, "ObjectMapper may not be null");
		Assert.notNull(docType, "docType may not be null");
		mapper = om;
		type = docType;
	}
	
	@Override
	public List<T> success(HttpResponse hr) throws Exception {
		JsonParser jp = mapper.getJsonFactory().createJsonParser(hr.getContent());
		if (jp.nextToken() != JsonToken.START_OBJECT) {
			throw new RuntimeException("Expected data to start with an Object");
		}
		
		Map<String, Integer> fields = readHeaderFields(jp);
		
		List<T> result;
		if (fields.containsKey(TOTAL_ROWS_FIELD_NAME)) {
			int totalRows = fields.get(TOTAL_ROWS_FIELD_NAME);
			if (totalRows == 0) {
				return Collections.emptyList();
			}
			result = new ArrayList<T>(totalRows);
		} else {
			result = new ArrayList<T>();
		}
		
		ParseState state = new ParseState();
		
		T first = parseFirstRow(jp, state);
		if (first == null) {
			return Collections.emptyList();
		} else {
			result.add(first);
		}
		
		while (jp.getCurrentToken() != null) {
			skipToField(jp, state.docFieldName, state);
			if (atEndOfRows(jp)) {
				return result;
			}
			result.add(jp.readValueAs(type));
			endRow(jp, state);
		}
		return result;
	}

	private T parseFirstRow(JsonParser jp, ParseState state)
			throws JsonParseException, IOException, JsonProcessingException,
			JsonMappingException {
		skipToField(jp, VALUE_FIELD_NAME, state);
		JsonNode value = null;
		if (atObjectStart(jp)) {
			value = jp.readValueAsTree();
			jp.nextToken();
			if (isEndOfRow(jp)) {
				state.docFieldName = VALUE_FIELD_NAME;
				T doc = mapper.readValue(value, type);
				endRow(jp, state);
				return doc;
			}
		}
		skipToField(jp, INCLUDED_DOC_FIELD_NAME, state);
		if (atObjectStart(jp)) {
			state.docFieldName = INCLUDED_DOC_FIELD_NAME;
			T doc = jp.readValueAs(type);
			endRow(jp, state);
			return doc;
		}
		return null;
	}

	private boolean isEndOfRow(JsonParser jp) {
		return jp.getCurrentToken() == JsonToken.END_OBJECT;
	}

	private void endRow(JsonParser jp, ParseState state) throws IOException, JsonParseException {
		state.inRow = false;
		jp.nextToken();
	}

	private boolean atObjectStart(JsonParser jp) {
		return jp.getCurrentToken() == JsonToken.START_OBJECT;
	}

	private boolean atEndOfRows(JsonParser jp) {
		return jp.getCurrentToken() != JsonToken.START_OBJECT;
	}
	
	private void skipToField(JsonParser jp, String fieldName, ParseState state) throws JsonParseException, IOException {
		String lastFieldName = null;
		while (jp.getCurrentToken() != null) {
			switch (jp.getCurrentToken()) {
			case FIELD_NAME:
				lastFieldName = jp.getCurrentName();
				jp.nextToken();
				break;
			case START_OBJECT:
				if (!state.inRow) {
					state.inRow = true;	
					jp.nextToken();
				} else {
					if (isInField(fieldName, lastFieldName)) {
						return;
					} else {
						jp.skipChildren();
					}	
				}
				break;
			default:
				if (isInField("key", lastFieldName)) {
					state.lastKey = jp.readValueAsTree(); 
				} else if (isInField("error", lastFieldName)) {
					JsonNode error = jp.readValueAsTree();
					throw new ViewResultException(state.lastKey, error.getValueAsText());
				} else if (isInField(fieldName, lastFieldName)) {
					jp.nextToken();
					return;
				}
				jp.nextToken();
				break;
			}
		}
	}

	private boolean isInField(String fieldName, String lastFieldName) {
		return lastFieldName != null && lastFieldName.equals(fieldName);
	}

	private Map<String, Integer> readHeaderFields(JsonParser jp) throws JsonParseException, IOException {
		Map<String, Integer> map = new HashMap<String, Integer>();
		jp.nextToken();
		String nextFieldName = jp.getCurrentName();
		while(!nextFieldName.equals(ROWS_FIELD_NAME)) {
			jp.nextToken();
			map.put(nextFieldName, Integer.valueOf(jp.getIntValue()));
			jp.nextToken();
			nextFieldName = jp.getCurrentName();
		}
		return map;
	}
	
	private static class ParseState {
		boolean inRow;
		JsonNode lastKey;
		String docFieldName = "";
	}
}
