package org.ektorp.support;

import java.io.*;
import java.lang.ref.*;
import java.lang.reflect.*;
import java.util.*;

import org.codehaus.jackson.map.*;
import org.ektorp.util.*;

public class SimpleViewGenerator {

	private final static String LOOKUP_BY_PROPERTY_TEMPLATE = "function(doc) { if(doc.%s) {emit(doc.%s, doc._id)} }";
	private final static String LOOKUP_BY_ITERABLE_PROPERTY_TEMPLATE = "function(doc) {for (var i in doc.%s) {emit(doc.%s[i], doc._id);}}";
	private SoftReference<ObjectMapper> mapperRef;
	
	public DesignDocument.View generateFindByView(String propertyName) {
		return new DesignDocument.View(String.format(LOOKUP_BY_PROPERTY_TEMPLATE, propertyName, propertyName));
	}
	
	public DesignDocument.View generateFindByIterableView(String propertyName) {
		return new DesignDocument.View(String.format(LOOKUP_BY_ITERABLE_PROPERTY_TEMPLATE, propertyName, propertyName));
	}
	
	public Map<String, DesignDocument.View> generateViews(final Object repository) {
		final Map<String, DesignDocument.View> views = new HashMap<String, DesignDocument.View>();
		final Class<?> repositoryClass = repository.getClass();
		
		final Class<?> handledType = repository instanceof CouchDbRepositorySupport<?> ? ((CouchDbRepositorySupport<?>) repository).getHandledType() : null;

		ReflectionUtils.eachAnnotation(repositoryClass, Views.class, new Predicate<Views>() {

			public boolean apply(Views input) {
				for (View v : input.value()) {
					addView(views, v, repositoryClass);
				}
				return true;
			}
		});
		
		ReflectionUtils.eachAnnotation(repositoryClass, View.class, new Predicate<View>() {

			public boolean apply(View input) {
				addView(views, input, repositoryClass);
				return true;
			}
		});
		
		ReflectionUtils.eachAnnotatedMethod(repositoryClass, GenerateView.class, new Predicate<Method>() {

			public boolean apply(Method input) {
				generateView(views, input, handledType);
				return true;
			}
		});
		return views;
	}

	private void addView(Map<String, DesignDocument.View> views, View input, Class<?> repositoryClass) {
		if (input.file().length() > 0) {
			views.put(input.name(), loadViewFromFile(views, input, repositoryClass));
		} else {
			views.put(input.name(), DesignDocument.View.of(input));
		}
	}

	private DesignDocument.View loadViewFromFile(Map<String, DesignDocument.View> views,
			View input, Class<?> repositoryClass) {
		try {
			InputStream in = repositoryClass.getResourceAsStream(input.file());
			if (in == null) {
				throw new FileNotFoundException("Could not load view file with path: " + input.file());
			}
			return mapper().readValue(in, DesignDocument.View.class);
		} catch (Exception e) {
			throw Exceptions.propagate(e);
		}
	}
	
	private boolean isIterable(Class<?> type) {
		return Iterable.class.isAssignableFrom(type);
	}
	
	private void generateView(Map<String, DesignDocument.View> views, Method me, Class<?> handledType) {
		String name = me.getName();
		if (!name.startsWith("findBy")) {
			throw new ViewGenerationException("Method annotated with GenerateView does not conform to the naming convention of 'findByXxxx'");
		}
		
		Class<?> type = resolveReturnType(me);
		if (type == null) {
			if (handledType != null) {
				type = handledType;
			} else {
				throw new ViewGenerationException("Could not resolve return type for method %s", me.getName());
			}
		}
		
		String finderName = name.substring(6);
		String fieldName = resolveFieldName(me ,finderName);
		Method getter = ReflectionUtils.findMethod(type, "get" + fieldName);
		if (getter == null) {
			// try pluralis
			fieldName += "s";
			getter = ReflectionUtils.findMethod(type, "get" + fieldName);
		}
		if (getter == null) {
			throw new ViewGenerationException("Could not generate view for method %s. No get method found for property %s in %s", name, name.substring(6), type);
		}
		
		fieldName = firstCharToLowerCase(fieldName);
		
		DesignDocument.View view;
		if (isIterable(getter.getReturnType())) {
			view = generateFindByIterableView(fieldName);
		} else {
			view = generateFindByView(fieldName);
		}
		
		views.put("by_" + firstCharToLowerCase(finderName), view);
	}

	private String resolveFieldName(Method me, String finderName) {
		GenerateView g = me.getAnnotation(GenerateView.class);
		String field = g.field();
		return field.length() == 0 ? finderName : g.field();
	}

	private String firstCharToLowerCase(String name) {
		return Character.toString(Character.toLowerCase(name.charAt(0))) + name.substring(1);
	}
	
	private Class<?> resolveReturnType(Method me) {
		Type returnType = me.getGenericReturnType();

		if(returnType instanceof ParameterizedType){
		    ParameterizedType type = (ParameterizedType) returnType;
		    Type[] typeArguments = type.getActualTypeArguments();
		    for(Type typeArgument : typeArguments){
		    	if (typeArgument instanceof Class<?>) {
		    		return (Class<?>) typeArgument;	
		    	}
		    }
		    return null;
		}
		return (Class<?>) returnType;
	}
	
	private ObjectMapper mapper() {
		if (mapperRef == null) {
			mapperRef = new SoftReference<ObjectMapper>(new ObjectMapper());	
		}
		ObjectMapper mapper = mapperRef.get();
		if (mapper == null) {
			mapper = new ObjectMapper();
			mapperRef = new SoftReference<ObjectMapper>(mapper);
		}
		return mapper;
	}
	
}
