package org.elsfs.tool.core.text;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Objects;

/**
 * the is {@link ArrayUtils}
 *
 * @author zeng
 * @since 0.0.3
 */
public class ArrayUtils {
  /**
   * 数组是否为空
   *
   * @param <T> 数组元素类型
   * @param array 数组
   * @return 是否为空
   */
  public static <T> boolean isEmpty(T[] array) {
    return array == null || array.length == 0;
  }

  /**
   * 对象是否为数组对象
   *
   * @param obj 对象
   * @return 是否为数组对象，如果为{@code null} 返回false
   */
  public static boolean isArray(Object obj) {
    return null != obj && obj.getClass().isArray();
  }

  /**
   * 数组或集合转String
   *
   * @param obj 集合或数组对象
   * @return 数组字符串，与集合转字符串格式相同
   */
  public static String toString(Object obj) {
    if (null == obj) {
      return null;
    }

    if (obj instanceof long[]) {
      return Arrays.toString((long[]) obj);
    } else if (obj instanceof int[]) {
      return Arrays.toString((int[]) obj);
    } else if (obj instanceof short[]) {
      return Arrays.toString((short[]) obj);
    } else if (obj instanceof char[]) {
      return Arrays.toString((char[]) obj);
    } else if (obj instanceof byte[]) {
      return Arrays.toString((byte[]) obj);
    } else if (obj instanceof boolean[]) {
      return Arrays.toString((boolean[]) obj);
    } else if (obj instanceof float[]) {
      return Arrays.toString((float[]) obj);
    } else if (obj instanceof double[]) {
      return Arrays.toString((double[]) obj);
    } else if (ArrayUtils.isArray(obj)) {
      // 对象数组
      try {
        return Arrays.deepToString((Object[]) obj);
      } catch (Exception ignore) {
        // ignore
      }
    }

    return obj.toString();
  }

  /**
   * 将新元素添加到已有数组中<br>
   * 添加新元素会生成一个新的数组，不影响原数组
   *
   * @param <T> 数组元素类型
   * @param buffer 已有数组
   * @param newElements 新元素
   * @return 新数组
   */
  @SafeVarargs
  public static <T> T[] append(T[] buffer, T... newElements) {
    if (isEmpty(buffer)) {
      return newElements;
    }
    return insert(buffer, buffer.length, newElements);
  }

  /**
   * 将新元素插入到到已有数组中的某个位置<br>
   * 添加新元素会生成一个新的数组，不影响原数组<br>
   * 如果插入位置为为负数，从原数组从后向前计数，若大于原数组长度，则空白处用null填充
   *
   * @param <T> 数组元素类型
   * @param buffer 已有数组
   * @param index 插入位置，此位置为对应此位置元素之前的空档
   * @param newElements 新元素
   * @return 新数组
   * @since 4.0.8
   */
  @SuppressWarnings("unchecked")
  public static <T> T[] insert(T[] buffer, int index, T... newElements) {
    return (T[]) insert((Object) buffer, index, newElements);
  }

  /**
   * 将新元素插入到到已有数组中的某个位置<br>
   * 添加新元素会生成一个新的数组，不影响原数组<br>
   * 如果插入位置为为负数，从原数组从后向前计数，若大于原数组长度，则空白处用null填充
   *
   * @param <T> 数组元素类型
   * @param array 已有数组
   * @param index 插入位置，此位置为对应此位置元素之前的空档
   * @param newElements 新元素
   * @return 新数组
   * @since 4.0.8
   */
  @SuppressWarnings({"unchecked", "SuspiciousSystemArraycopy"})
  public static <T> Object insert(Object array, int index, T... newElements) {
    if (isEmpty(newElements)) {
      return array;
    }
    if (Objects.isNull(array)) {
      return newElements;
    }

    final int len = length(array);
    if (index < 0) {
      index = (index % len) + len;
    }

    // 已有数组的元素类型
    final Class<?> originComponentType = array.getClass().getComponentType();
    Object newEleArr = newElements;
    // 如果 已有数组的元素类型是 原始类型，则需要转换 新元素数组 为该类型，避免ArrayStoreException
    if (originComponentType.isPrimitive()) {
      // TODO 则需要转换
    }
    final Object result =
        Array.newInstance(originComponentType, Math.max(len, index) + newElements.length);
    System.arraycopy(array, 0, result, 0, Math.min(len, index));
    System.arraycopy(newEleArr, 0, result, index, newElements.length);
    if (index < len) {
      System.arraycopy(array, index, result, index + newElements.length, len - index);
    }
    return result;
  }

  /**
   * 获取数组长度<br>
   * 如果参数为{@code null}，返回0
   *
   * <pre>
   * ArrayUtil.length(null)            = 0
   * ArrayUtil.length([])              = 0
   * ArrayUtil.length([null])          = 1
   * ArrayUtil.length([true, false])   = 2
   * ArrayUtil.length([1, 2, 3])       = 3
   * ArrayUtil.length(["a", "b", "c"]) = 3
   * </pre>
   *
   * @param array 数组对象
   * @return 数组长度
   * @throws IllegalArgumentException 如果参数不为数组，抛出此异常
   * @see Array#getLength(Object)
   */
  public static int length(Object array) throws IllegalArgumentException {
    if (null == array) {
      return 0;
    }
    return Array.getLength(array);
  }
}
