package org.elsfs.tool.core.text;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * the is {@link StrUtils}
 *
 * @author zeng
 * @since 0.0.3
 */
public class StrUtils extends StringUtils {

  /**
   * 将对象转为字符串<br>
   *
   * <pre>
   * 1、Byte数组和ByteBuffer会被转换为对应字符串的数组
   * 2、对象数组会调用Arrays.toString方法
   * </pre>
   *
   * @param obj 对象
   * @return 字符串
   */
  public static String utf8Str(Object obj) {
    return str(obj, StandardCharsets.UTF_8);
  }

  /**
   * 将对象转为字符串
   *
   * <pre>
   * 1、Byte数组和ByteBuffer会被转换为对应字符串的数组
   * 2、对象数组会调用Arrays.toString方法
   * </pre>
   *
   * @param obj 对象
   * @param charsetName 字符集
   * @return 字符串
   * @deprecated 请使用 {@link #str(Object, Charset)}
   */
  @Deprecated
  public static String str(Object obj, String charsetName) {
    return str(obj, Charset.forName(charsetName));
  }

  /**
   * 将对象转为字符串
   *
   * <pre>
   * <li>1、Byte数组和ByteBuffer会被转换为对应字符串的数组</li>
   * 2、对象数组会调用Arrays.toString方法
   * </pre>
   *
   * @param obj 对象
   * @param charset 字符集
   * @return 字符串
   */
  public static String str(Object obj, Charset charset) {
    if (null == obj) {
      return null;
    }

    if (obj instanceof String) {
      return (String) obj;
    } else if (obj instanceof byte[]) {
      return str((byte[]) obj, charset);
    } else if (obj instanceof Byte[]) {
      return str((Byte[]) obj, charset);
    } else if (obj instanceof ByteBuffer) {
      return str((ByteBuffer) obj, charset);
    } else if (ArrayUtils.isArray(obj)) {
      return ArrayUtils.toString(obj);
    }

    return obj.toString();
  }

  /**
   * 解码字节码
   *
   * @param data 字符串
   * @param charset 字符集，如果此字段为空，则解码的结果取决于平台
   * @return 解码后的字符串
   */
  public static String str(byte[] data, Charset charset) {
    if (data == null) {
      return null;
    }

    if (null == charset) {
      return new String(data);
    }
    return new String(data, charset);
  }

  /**
   * 解码字节码
   *
   * @param data 字符串
   * @param charset 字符集，如果此字段为空，则解码的结果取决于平台
   * @return 解码后的字符串
   */
  public static String str(Byte[] data, Charset charset) {
    if (data == null) {
      return null;
    }

    byte[] bytes = new byte[data.length];
    Byte dataByte;
    for (int i = 0; i < data.length; i++) {
      dataByte = data[i];
      bytes[i] = (null == dataByte) ? -1 : dataByte;
    }

    return str(bytes, charset);
  }

  /**
   * 将编码的byteBuffer数据转换为字符串
   *
   * @param data 数据
   * @param charset 字符集，如果为空使用当前系统字符集
   * @return 字符串
   */
  public static String str(ByteBuffer data, Charset charset) {
    if (null == charset) {
      charset = Charset.defaultCharset();
    }
    return charset.decode(data).toString();
  }
}
