package org.elsfs.tool.core.util;

import org.elsfs.tool.core.text.CharUtil;

/**
 * {@link CharSequence} 相关工具类封装
 *
 * @author zeng
 * @since 0.0.3
 */
public class CharSequenceUtil {

  /** 字符串常量：空字符串 {@code ""} */
  public static final String EMPTY = "";

  /**
   * 字符串是否为空白，空白的定义如下：
   *
   * <ol>
   *   <li>{@code null}
   *   <li>空字符串：{@code ""}
   *   <li>空格、全角空格、制表符、换行符，等不可见字符
   * </ol>
   *
   * <p>例：
   *
   * <ul>
   *   <li>{@code StrUtil.isBlank(null) // true}
   *   <li>{@code StrUtil.isBlank("") // true}
   *   <li>{@code StrUtil.isBlank(" \t\n") // true}
   *   <li>{@code StrUtil.isBlank("abc") // false}
   * </ul>
   *
   * <p>注意：该方法与 {@link #isEmpty(CharSequence)} 的区别是： 该方法会校验空白字符，且性能相对于 {@link
   * #isEmpty(CharSequence)} 略慢。 <br>
   *
   * <p>建议：
   *
   * <ul>
   *   <li>该方法建议仅对于客户端（或第三方接口）传入的参数使用该方法。
   * </ul>
   *
   * @param str 被检测的字符串
   * @return 若为空白，则返回 true
   * @see #isEmpty(CharSequence)
   */
  public static boolean isBlank(CharSequence str) {
    final int length;
    if ((str == null) || ((length = str.length()) == 0)) {
      return true;
    }

    for (int i = 0; i < length; i++) {
      // 只要有一个非空字符即为非空字符串
      if (!CharUtil.isBlankChar(str.charAt(i))) {
        return false;
      }
    }

    return true;
  }

  /**
   * 字符串是否为空，空的定义如下：
   *
   * <ol>
   *   <li>{@code null}
   *   <li>空字符串：{@code ""}
   * </ol>
   *
   * <p>例：
   *
   * <ul>
   *   <li>{@code StrUtil.isEmpty(null) // true}
   *   <li>{@code StrUtil.isEmpty("") // true}
   *   <li>{@code StrUtil.isEmpty(" \t\n") // false}
   *   <li>{@code StrUtil.isEmpty("abc") // false}
   * </ul>
   *
   * <p>注意：该方法与 {@link #isBlank(CharSequence)} 的区别是：该方法不校验空白字符。
   *
   * <p>建议：
   *
   * <ul>
   *   <li>该方法建议用于工具类或任何可以预期的方法参数的校验中。
   * </ul>
   *
   * @param str 被检测的字符串
   * @return 是否为空
   */
  public static boolean isEmpty(CharSequence str) {
    return str == null || str.length() == 0;
  }

  /**
   * 大写首字母<br>
   * 例如：str = name, return Name
   *
   * @param str 字符串
   * @return 字符串
   */
  public static String upperFirst(CharSequence str) {
    if (null == str) {
      return null;
    }
    if (str.length() > 0) {
      char firstChar = str.charAt(0);
      if (Character.isLowerCase(firstChar)) {
        return Character.toUpperCase(firstChar) + subSuf(str, 1);
      }
    }
    return str.toString();
  }

  /**
   * 切割指定位置之后部分的字符串
   *
   * @param string 字符串
   * @param fromIndex 切割开始的位置（包括）
   * @return 切割后后剩余的后半部分字符串
   */
  public static String subSuf(CharSequence string, int fromIndex) {
    if (isEmpty(string)) {
      return null;
    }
    return sub(string, fromIndex, string.length());
  }

  /**
   * 改进JDK subString<br>
   * index从0开始计算，最后一个字符为-1<br>
   * 如果from和to位置一样，返回 "" <br>
   * 如果from或to为负数，则按照length从后向前数位置，如果绝对值大于字符串长度，则from归到0，to归到length<br>
   * 如果经过修正的index中from大于to，则互换from和to example: <br>
   * abcdefgh 2 3 =》 c <br>
   * abcdefgh 2 -3 =》 cde <br>
   *
   * @param str String
   * @param fromIndexInclude 开始的index（包括）
   * @param toIndexExclude 结束的index（不包括）
   * @return 字串
   */
  public static String sub(CharSequence str, int fromIndexInclude, int toIndexExclude) {
    if (isEmpty(str)) {
      return str(str);
    }
    int len = str.length();

    if (fromIndexInclude < 0) {
      fromIndexInclude = len + fromIndexInclude;
      if (fromIndexInclude < 0) {
        fromIndexInclude = 0;
      }
    } else if (fromIndexInclude > len) {
      fromIndexInclude = len;
    }

    if (toIndexExclude < 0) {
      toIndexExclude = len + toIndexExclude;
      if (toIndexExclude < 0) {
        toIndexExclude = len;
      }
    } else if (toIndexExclude > len) {
      toIndexExclude = len;
    }

    if (toIndexExclude < fromIndexInclude) {
      int tmp = fromIndexInclude;
      fromIndexInclude = toIndexExclude;
      toIndexExclude = tmp;
    }

    if (fromIndexInclude == toIndexExclude) {
      return EMPTY;
    }

    return str.toString().substring(fromIndexInclude, toIndexExclude);
  }

  /**
   * {@link CharSequence} 转为字符串，null安全
   *
   * @param cs {@link CharSequence}
   * @return 字符串
   */
  public static String str(CharSequence cs) {
    return null == cs ? null : cs.toString();
  }
}
