package org.elsfs.tool.sql.mybatisplus.conditions;

import com.baomidou.mybatisplus.core.conditions.AbstractWrapper;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.interfaces.Join;
import com.baomidou.mybatisplus.core.conditions.interfaces.Nested;
import java.util.Collection;
import java.util.function.Consumer;
import org.elsfs.tool.core.text.NamingCase;
import org.elsfs.tool.core.text.StrFormatter;
import org.elsfs.tool.sql.conditions.Condition;

/**
 * the is {@link AbstractMybatisPlusCondition}
 *
 * @author zeng
 * @since 0.0.3
 */
public abstract class AbstractMybatisPlusCondition<Children, E> implements Condition<Children> {

  /** 子类引用 */
  @SuppressWarnings("unchecked")
  protected final Children childThis = (Children) this;

  /** 条件包装器 */
  protected final AbstractWrapper<E, String, ?> wrapper;

  protected AbstractMybatisPlusCondition(AbstractWrapper<E, String, ?> wrapper) {
    this.wrapper = wrapper;
  }

  /**
   * 获取操作Action实例
   *
   * @param wrapper 条件Wrapper对象
   * @return Action实例
   */
  protected abstract Children instance(Wrapper<E> wrapper);

  /**
   * 获取JOIN对象
   *
   * @return JOIN对象
   */
  protected Join<?> getJoin() {
    return wrapper;
  }

  /**
   * 获取JOIN对象
   *
   * @return JOIN对象
   */
  @SuppressWarnings("unchecked")
  protected Nested<Wrapper<E>, Wrapper<E>> getNested() {
    return (Nested<Wrapper<E>, Wrapper<E>>) wrapper;
  }

  /**
   * 转换为数据库字段名称
   *
   * @param field 实体字段名称
   * @return 数据库字段名称
   */
  protected String convertToDbFieldName(String field) {
    return NamingCase.toUnderlineCase(field);
  }

  /**
   * 等于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public Children eq(boolean condition, String field, Object value) {
    wrapper.eq(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 字段 = ( sql语句 )
   *
   * <p>例1: eqSql("id", "select id from table where name = 'JunJun'")
   *
   * @param condition 条件
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public Children eqSql(boolean condition, String field, String sqlValue) {
    if (condition) {
      wrapper.apply(StrFormatter.format("{}=({})", convertToDbFieldName(field), sqlValue));
    }
    return childThis;
  }

  /**
   * AND 嵌套
   *
   * <p>例: and(i -&gt; i.eq("name", "李白").ne("status", "活着"))
   *
   * @param condition 执行条件
   * @param consumer 消费函数
   * @return children
   */
  @Override
  public Children and(boolean condition, Consumer<Children> consumer) {
    if (condition) {
      getNested()
          .and(
              wp -> {
                Children child = instance(wp);
                consumer.accept(child);
              });
    }
    return childThis;
  }

  /**
   * 应用action消费器
   *
   * @param condition 条件
   * @param actionConsumer action消费器
   * @return 具体实现
   */
  @Override
  public Children apply(boolean condition, Consumer<Children> actionConsumer) {
    if (condition) {
      actionConsumer.accept(childThis);
    }
    return childThis;
  }

  /**
   * 范围
   *
   * @param condition 是否应用条件
   * @param field 字段名
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public Children between(boolean condition, String field, Object leftValue, Object rightValue) {
    if (condition) {
      wrapper.between(convertToDbFieldName(field), leftValue, rightValue);
    }
    return childThis;
  }

  @Override
  public Children exists(boolean condition, String sqlValue, Object... values) {
    getJoin().exists(condition, sqlValue, values);
    return childThis;
  }

  @Override
  public Children expression(boolean condition, String sqlValue, Object... values) {
    getJoin().apply(condition, sqlValue, values);
    return childThis;
  }

  @Override
  public Children ge(boolean condition, String field, Object value) {
    wrapper.ge(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children geSql(boolean condition, String field, String sqlValue) {
    wrapper.geSql(condition, convertToDbFieldName(field), sqlValue);
    return childThis;
  }

  @Override
  public Children gt(boolean condition, String field, Object value) {
    wrapper.gt(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children in(boolean condition, String field, Collection<?> values) {
    wrapper.in(condition, convertToDbFieldName(field), values);
    return childThis;
  }

  @Override
  public Children in(boolean condition, String field, Object... values) {
    wrapper.in(condition, convertToDbFieldName(field), values);
    return childThis;
  }

  @Override
  public Children isNotNull(boolean condition, String field) {
    wrapper.isNotNull(condition, convertToDbFieldName(field));
    return childThis;
  }

  @Override
  public Children inSql(boolean condition, String field, String sqlValue) {
    wrapper.inSql(condition, convertToDbFieldName(field), sqlValue);
    return childThis;
  }

  @Override
  public Children gtSql(boolean condition, String field, String sqlValue) {
    wrapper.gtSql(condition, convertToDbFieldName(field), sqlValue);
    return childThis;
  }

  @Override
  public Children le(boolean condition, String field, Object value) {
    wrapper.le(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children leSql(boolean condition, String field, String sqlValue) {
    wrapper.leSql(condition, convertToDbFieldName(field), sqlValue);
    return childThis;
  }

  @Override
  public Children like(boolean condition, String field, Object value) {
    wrapper.like(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children likeLeft(boolean condition, String field, Object value) {
    wrapper.likeLeft(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children likeRight(boolean condition, String field, Object value) {
    wrapper.likeRight(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children lt(boolean condition, String field, Object value) {
    wrapper.lt(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children ltSql(boolean condition, String field, String sqlValue) {
    wrapper.ltSql(condition, convertToDbFieldName(field), sqlValue);
    return childThis;
  }

  @Override
  public Children ne(boolean condition, String field, Object value) {
    wrapper.ne(condition, convertToDbFieldName(field), value);
    return childThis;
  }

  @Override
  public Children nested(boolean condition, Consumer<Children> consumer) {
    if (condition) {
      getNested()
          .nested(
              wp -> {
                Children child = instance(wp);
                consumer.accept(child);
              });
    }
    return childThis;
  }

  @Override
  public Children notBetween(boolean condition, String field, Object leftValue, Object rightValue) {
    wrapper.notBetween(condition, convertToDbFieldName(field), leftValue, rightValue);
    return childThis;
  }

  @Override
  public Children isNull(boolean condition, String field) {
    wrapper.isNull(condition, convertToDbFieldName(field));
    return childThis;
  }

  @Override
  public Children notExists(boolean condition, String sqlValue, Object... values) {
    getJoin().notExists(condition, sqlValue, values);
    return childThis;
  }

  @Override
  public Children notIn(boolean condition, String field, Collection<?> values) {
    this.wrapper.notIn(condition, this.convertToDbFieldName(field), values);
    return childThis;
  }

  @Override
  public Children notIn(boolean condition, String field, Object... values) {
    this.wrapper.notIn(condition, this.convertToDbFieldName(field), values);
    return childThis;
  }

  @Override
  public Children notInSql(boolean condition, String field, String sqlValue) {
    this.wrapper.notInSql(condition, this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  @Override
  public Children notLike(boolean condition, String field, Object value) {
    this.wrapper.notLike(condition, this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 拼接 OR
   *
   * @param condition 执行条件
   * @return children
   */
  @Override
  public Children or(boolean condition) {
    if (condition) {
      this.or();
    }
    return childThis;
  }

  /**
   * 拼接 OR
   *
   * @return children
   */
  @Override
  public Children or() {
    this.getJoin().or();
    return this.childThis;
  }

  @Override
  public Children or(boolean condition, Consumer<Children> consumer) {
    if (condition) {
      return this.or(consumer);
    }
    return this.childThis;
  }
}
