package org.elsfs.tool.sql.mybatisplus.mybatis;

import com.baomidou.mybatisplus.core.conditions.AbstractWrapper;
import com.baomidou.mybatisplus.core.conditions.query.Query;
import java.util.Arrays;
import java.util.List;
import org.elsfs.tool.core.page.IPage;
import org.elsfs.tool.sql.singular.facade.QueryOperations;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 抽象查询操作实现
 *
 * @param <C> 子类实现类型
 * @param <E> 实体类型
 * @author zeng
 * @since 0.0.4
 */
public abstract class AbstractMybatisPlusQueryOperations<C extends QueryOperations<C>, E>
    extends AbstractMybatisPlusConditionOperations<C, E> implements QueryOperations<C> {

  /** 分页对象 */
  protected IPage<?> page;

  protected AbstractMybatisPlusQueryOperations(AbstractWrapper<E, String, ?> queryWrapper) {
    super(queryWrapper);
  }

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return 具体实现
   */
  @Override
  public C select(boolean condition, String... fields) {
    if (condition) {
      return this.select(fields);
    }

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return 具体实现
   */
  @Override
  public C select(boolean condition, Enum<?>... fields) {
    if (condition) {
      return this.select(fields);
    }

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param fields 字段数组
   * @return 具体实现
   */
  @Override
  public final C select(String... fields) {
    this.query()
        .select(Arrays.stream(fields).map(this::convertToDbFieldName).toArray(String[]::new));

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param fields 字段枚举数组
   * @return 具体实现
   */
  @Override
  public C select(Enum<?>... fields) {
    this.query()
        .select(
            Arrays.stream(fields)
                .map(this::resolveFieldName)
                .map(this::convertToDbFieldName)
                .toArray(String[]::new));

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  @Override
  public C selectAs(boolean condition, Enum<?> field, String fieldAlias) {
    if (condition) {
      return this.selectAs(field, fieldAlias);
    }

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  @Override
  public C selectAs(boolean condition, Enum<?> field, Enum<?> fieldAlias) {
    if (condition) {
      return this.selectAs(field, fieldAlias);
    }

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param field 字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  @Override
  public C selectAs(Enum<?> field, String fieldAlias) {
    this.query()
        .select(
            this.appendFieldAlias(
                this.convertToDbFieldName(this.resolveFieldName(field)),
                this.convertToDbFieldName(fieldAlias)));

    return this.childThis;
  }

  /**
   * 选择字段
   *
   * @param field 字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  @Override
  public C selectAs(Enum<?> field, Enum<?> fieldAlias) {
    return this.selectAs(field, fieldAlias.name());
  }

  /**
   * 设置分页信息
   *
   * @param pageable 可分页对象
   * @return 具体实现
   */
  @Override
  public C pageable(IPage<?> pageable) {
    this.page = pageable;
    return this.childThis;
  }

  /**
   * 设置分页信息
   *
   * @param condition 执行条件
   * @param pageable 可分页对象
   * @return 具体实现
   */
  @Override
  public C pageable(boolean condition, IPage<?> pageable) {
    if (condition) {
      this.page = pageable;
    }

    return this.childThis;
  }

  /**
   * 偏移数量
   *
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C offset(long num) {
    this.wrapper.last("OFFSET " + num);
    return this.childThis;
  }

  /**
   * 偏移数量
   *
   * @param condition 执行条件
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C offset(boolean condition, long num) {
    if (condition) {
      return this.offset(num);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param condition 执行条件
   * @param field 单个字段
   * @return children
   */
  @Override
  public C groupBy(boolean condition, String field) {
    if (condition) {
      return this.groupBy(field);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C groupBy(List<String> fields) {
    this.wrapper.groupBy(fields.stream().map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param condition 执行条件
   * @param field 单个字段枚举
   * @return children
   */
  @Override
  public C groupBy(boolean condition, Enum<?> field) {
    if (condition) {
      return this.groupBy(field);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C groupBy(boolean condition, String... fields) {
    if (condition) {
      return this.groupBy(fields);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C groupBy(boolean condition, List<String> fields) {
    if (condition) {
      return this.groupBy(fields);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C groupBy(boolean condition, Enum<?>... fields) {
    if (condition) {
      return this.groupBy(fields);
    }

    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param field 单个字段
   * @return children
   */
  @Override
  public C groupBy(String field) {
    this.wrapper.groupBy(this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C groupBy(String... fields) {
    this.wrapper.groupBy(Arrays.stream(fields).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C groupBy(Enum<?>... fields) {
    this.wrapper.groupBy(
        Arrays.stream(fields).map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param condition 执行条件
   * @param field 单个字段枚举
   * @return children
   */
  @Override
  public C orderByAsc(boolean condition, Enum<?> field) {
    if (condition) {
      return this.orderByAsc(field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C orderByAsc(boolean condition, List<String> fields) {
    if (condition) {
      return this.orderByAsc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param condition 执行条件
   * @param field 单个字段
   * @return children
   */
  @Override
  public C orderByAsc(boolean condition, String field) {
    if (condition) {
      return this.orderByAsc(field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C orderByAsc(boolean condition, String... fields) {
    if (condition) {
      return this.orderByAsc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C orderByAsc(boolean condition, Enum<?>... fields) {
    if (condition) {
      return this.orderByAsc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param field 单个字段
   * @return children
   */
  @Override
  public C orderByAsc(String field) {
    this.wrapper.orderByAsc(this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C orderByAsc(String... fields) {
    this.wrapper.orderByAsc(Arrays.stream(fields).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  @Override
  public C orderByAsc(List<String> fields) {
    this.wrapper.orderByAsc(fields.stream().map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C orderByAsc(Enum<?>... fields) {
    this.wrapper.orderByAsc(
        Arrays.stream(fields).map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C groupByEnum(List<Enum<?>> fields) {
    this.wrapper.groupBy(
        fields.stream().map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C groupByEnum(boolean condition, List<Enum<?>> fields) {
    if (condition) {
      return this.groupByEnum(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param condition 执行条件
   * @param field 字段
   * @return children
   */
  @Override
  public C orderByDesc(boolean condition, String field) {
    if (condition) {
      return this.orderByDesc(field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @return children
   */
  @Override
  public C orderByDesc(boolean condition, Enum<?> field) {
    if (condition) {
      return this.orderByDesc(field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByDesc(boolean condition, Enum<?>... fields) {
    if (condition) {
      return this.orderByDesc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderByDesc(boolean condition, String... fields) {
    if (condition) {
      return this.orderByDesc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param field 字段
   * @return children
   */
  @Override
  public C orderByDesc(String field) {
    this.wrapper.orderByDesc(this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderByDesc(List<String> fields) {
    this.wrapper.orderByDesc(fields.stream().map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderByDesc(String... fields) {
    this.wrapper.orderByDesc(Arrays.stream(fields).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderByDesc(boolean condition, List<String> fields) {
    if (condition) {
      this.orderByDesc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByDesc(Enum<?>... fields) {
    this.wrapper.orderByDesc(
        Arrays.stream(fields).map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C orderByEnumAsc(List<Enum<?>> fields) {
    this.wrapper.orderByAsc(
        fields.stream().map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  @Override
  public C orderByEnumAsc(boolean condition, List<Enum<?>> fields) {
    if (condition) {
      return this.orderByEnumAsc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByEnumDesc(boolean condition, List<Enum<?>> fields) {
    if (condition) {
      return this.orderByEnumDesc(fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByEnumDesc(List<Enum<?>> fields) {
    this.wrapper.orderByDesc(
        fields.stream().map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段
   * @return children
   */
  @Override
  public C orderBy(boolean condition, boolean isAsc, String field) {
    if (condition) {
      return this.orderBy(isAsc, field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段枚举
   * @return children
   */
  @Override
  public C orderBy(boolean condition, boolean isAsc, Enum<?> field) {
    if (condition) {
      return this.orderBy(isAsc, field);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderBy(boolean condition, boolean isAsc, List<String> fields) {
    if (condition) {
      return this.orderBy(isAsc, fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderBy(boolean condition, boolean isAsc, String... fields) {
    if (condition) {
      return this.orderBy(isAsc, fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段
   * @return children
   */
  @Override
  public C orderBy(boolean isAsc, String field) {
    this.wrapper.orderBy(true, isAsc, this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderBy(boolean condition, boolean isAsc, Enum<?>... fields) {
    if (condition) {
      return this.orderBy(isAsc, fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderBy(boolean isAsc, List<String> fields) {
    this.wrapper.orderBy(true, isAsc, fields.stream().map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  @Override
  public C orderBy(boolean isAsc, String... fields) {
    this.wrapper.orderBy(
        true, isAsc, Arrays.stream(fields).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderBy(boolean isAsc, Enum<?>... fields) {
    this.wrapper.orderBy(
        true,
        isAsc,
        Arrays.stream(fields).map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByEnum(boolean condition, boolean isAsc, List<Enum<?>> fields) {
    if (condition) {
      return this.orderByEnum(isAsc, fields);
    }

    return this.childThis;
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  @Override
  public C orderByEnum(boolean isAsc, List<Enum<?>> fields) {
    this.wrapper.orderBy(
        true,
        isAsc,
        fields.stream().map(this::resolveFieldName).map(this::convertToDbFieldName).toList());
    return this.childThis;
  }

  /**
   * HAVING ( sql语句 )
   *
   * <p>例1: having("sum(age) &gt; 10")
   *
   * <p>例2: having("sum(age) &gt; {0}", 10)
   *
   * @param condition 执行条件
   * @param sqlValue sql 语句
   * @param params 参数数组
   * @return children
   */
  @Override
  public C having(boolean condition, String sqlValue, Object... params) {
    if (condition) {
      return this.having(sqlValue, params);
    }

    return this.childThis;
  }

  /**
   * HAVING ( sql语句 )
   *
   * <p>例1: having("sum(age) &gt; 10")
   *
   * <p>例2: having("sum(age) &gt; {0}", 10)
   *
   * @param sqlValue sql 语句
   * @param params 参数数组
   * @return children
   */
  @Override
  public C having(String sqlValue, Object... params) {
    this.wrapper.having(sqlValue, params);
    return this.childThis;
  }

  /**
   * 获取Query查询对象
   *
   * @return 查询对象
   */
  protected abstract Query<?, E, String> query();

  /**
   * 添加字段别名
   *
   * @param field 字段名称
   * @param fieldAlias 字段别名
   * @return 带别名的字段名称
   */
  protected String appendFieldAlias(String field, String fieldAlias) {
    return field + SqlPool.AS_KEYWORD_WITH_SPACE + fieldAlias;
  }

  /**
   * 限制查询数量
   *
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C limit(long num) {
    this.wrapper.last("LIMIT " + num);
    return this.childThis;
  }

  /**
   * 限制查询数据
   *
   * @param offset 开始位置
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C limit(long offset, long num) {
    this.wrapper.last("LIMIT " + num + " OFFSET " + offset);
    return this.childThis;
  }

  /**
   * 限制查询数量
   *
   * @param condition 执行条件
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C limit(boolean condition, long num) {
    if (condition) {
      return this.limit(num);
    }

    return this.childThis;
  }

  /**
   * 限制查询数据
   *
   * @param condition 执行条件
   * @param offset 开始位置
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public C limit(boolean condition, long offset, long num) {
    if (condition) {
      return this.limit(offset, num);
    }

    return this.childThis;
  }
}
