package org.elsfs.tool.sql.mybatisplus.mybatis;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.update.Update;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import org.elsfs.tool.core.text.NamingCase;
import org.elsfs.tool.sql.exception.SqlBuilderException;
import org.elsfs.tool.sql.mybatisplus.mapper.ElsfsMapper;
import org.elsfs.tool.sql.singular.facade.UpdateAction;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * MybatisPlus更新动作实现
 *
 * @param <E> 实体类型
 * @author zeng
 * @since 0.0.4
 */
public class MybatisPlusUpdateAction<E>
    extends AbstractMybatisPlusConditionOperations<UpdateAction<Long>, E>
    implements UpdateAction<Long> {

  /** 基础Mapper */
  protected final ElsfsMapper<E> elsfsMapper;

  public MybatisPlusUpdateAction(UpdateWrapper<E> updateWrapper, ElsfsMapper<E> elsfsMapper) {
    super(updateWrapper);
    this.elsfsMapper = elsfsMapper;
  }

  /**
   * 获取Wrapper对象，用于高级查询
   *
   * @return Wrapper查询对象
   */
  @SuppressWarnings("unchecked")
  protected UpdateWrapper<E> getWrapper() {
    return (UpdateWrapper<E>) this.wrapper;
  }

  /**
   * 设置值
   *
   * @param field 表字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> set(String field, Object value) {
    this.getWrapper().set(field, value);
    return this;
  }

  /**
   * 设置值
   *
   * @param field 表字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> set(Enum<?> field, Object value) {
    return this.set(this.resolveFieldName(field), value);
  }

  /**
   * 设置值
   *
   * @param condition 是否应用设置
   * @param field 表字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> set(boolean condition, String field, Object value) {
    if (condition) {
      return this.set(field, value);
    }

    return this;
  }

  /**
   * 设置值
   *
   * @param condition 是否应用设置
   * @param field 表字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> set(boolean condition, Enum<?> field, Object value) {
    if (condition) {
      return this.set(field, value);
    }

    return this;
  }

  /**
   * 获取更新接口实现
   *
   * @return 实现对象
   */
  protected Update<?, ?> getUpdate() {
    return (Update<?, ?>) this.wrapper;
  }

  /**
   * 设置SQL
   *
   * @param sqlValue SQL片段：name=xxx
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> setSql(String sqlValue) {
    this.getUpdate().setSql(sqlValue);
    return this;
  }

  /**
   * 设置SQL
   *
   * @param condition 是否应用设置SQL
   * @param sqlValue SQL片段：name=xxx
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> setSql(boolean condition, String sqlValue) {
    if (condition) {
      return this.setSql(sqlValue);
    }

    return this;
  }

  /**
   * 执行动作
   *
   * @return 结果
   */
  @Override
  public Long execute() {
    if (this.elsfsMapper == null) {
      throw new SqlBuilderException("嵌套查询条件设置Action不支持execute操作");
    }
    return (long) this.elsfsMapper.update(null, this.wrapper);
  }

  /**
   * 解析字段枚举名称
   *
   * @param field 字段枚举
   * @return 字段名称
   */
  @Override
  public String resolveFieldName(Enum<?> field) {
    return NamingCase.toUnderlineCase(field.name());
  }

  /**
   * 解析字段枚举名称
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @return 字段名称
   */
  @Override
  public String resolveFieldName(String tableAlias, Enum<?> field) {
    return tableAlias + SqlPool.DOT + field.name();
  }

  /**
   * 获取操作Action实例
   *
   * @param wrapper 条件Wrapper对象
   * @return Action实例
   */
  @Override
  protected UpdateAction<Long> instance(Wrapper<E> wrapper) {
    if (wrapper instanceof UpdateWrapper<E> updateWrapper) {
      return new MybatisPlusUpdateAction<>(updateWrapper, null);
    }

    return new MybatisPlusUpdateAction<>(null, null);
  }

  /**
   * 限制查询数量
   *
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> limit(long num) {
    this.wrapper.last("LIMIT " + num);
    return this;
  }

  /**
   * 限制查询数量
   *
   * @param condition 执行条件
   * @param num 数量
   * @return 具体实现
   */
  @Override
  public UpdateAction<Long> limit(boolean condition, long num) {
    if (condition) {
      return this.limit(num);
    }

    return this;
  }
}
