package org.elsfs.tool.sql.mybatisplus.mybatis;

import com.baomidou.mybatisplus.core.conditions.AbstractWrapper;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.interfaces.Join;
import com.baomidou.mybatisplus.core.conditions.interfaces.Nested;
import com.baomidou.mybatisplus.core.toolkit.Assert;
import java.util.Collection;
import java.util.function.Consumer;
import org.elsfs.tool.core.text.NamingCase;
import org.elsfs.tool.core.text.StrFormatter;
import org.elsfs.tool.sql.mybatisplus.AbstractConditionOperations;

/**
 * 普通条件操作抽象实现
 *
 * @param <C> 子类实现类型
 * @param <E> 实体类型
 * @author zeng
 * @since 0.0.4
 */
public abstract class AbstractMybatisPlusConditionOperations<C, E>
    extends AbstractConditionOperations<C> {

  /** 条件包装器 */
  protected final AbstractWrapper<E, String, ?> wrapper;

  protected AbstractMybatisPlusConditionOperations(AbstractWrapper<E, String, ?> wrapper) {
    this.wrapper = wrapper;
  }

  /**
   * 获取JOIN对象
   *
   * @return JOIN对象
   */
  protected Join<?> getJoin() {
    return this.wrapper;
  }

  /**
   * 获取JOIN对象
   *
   * @return JOIN对象
   */
  @SuppressWarnings("unchecked")
  protected Nested<Wrapper<E>, Wrapper<E>> getNested() {
    return (Nested<Wrapper<E>, Wrapper<E>>) this.wrapper;
  }

  /**
   * 解析字段枚举名称
   *
   * @param field 字段枚举
   * @return 字段名称
   */
  public abstract String resolveFieldName(Enum<?> field);

  /**
   * 解析字段枚举名称
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @return 字段名称
   */
  public abstract String resolveFieldName(String tableAlias, Enum<?> field);

  /**
   * 转换为数据库字段名称
   *
   * @param field 实体字段名称
   * @return 数据库字段名称
   */
  protected String convertToDbFieldName(String field) {
    return NamingCase.toUnderlineCase(field);
  }

  /**
   * 等于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C eq(String field, Object value) {
    this.wrapper.eq(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 不等于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ne(String field, Object value) {
    this.wrapper.ne(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 不等于
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ne(Enum<?> field, Object value) {
    return this.ne(this.resolveFieldName(field), value);
  }

  /**
   * 不等于
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ne(String tableAlias, Enum<?> field, Object value) {
    return this.ne(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 大于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C gt(String field, Object value) {
    this.wrapper.gt(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 大于
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C gt(Enum<?> field, Object value) {
    return this.gt(this.resolveFieldName(field), value);
  }

  /**
   * 大于
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C gt(String tableAlias, Enum<?> field, Object value) {
    return this.gt(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 大于等于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ge(String field, Object value) {
    this.wrapper.ge(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 大于等于
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ge(Enum<?> field, Object value) {
    return this.ge(this.resolveFieldName(field), value);
  }

  /**
   * 大于等于
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C ge(String tableAlias, Enum<?> field, Object value) {
    return this.ge(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 小于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C lt(String field, Object value) {
    this.wrapper.lt(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 小于
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C lt(Enum<?> field, Object value) {
    return this.lt(this.resolveFieldName(field), value);
  }

  /**
   * 小于
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C lt(String tableAlias, Enum<?> field, Object value) {
    return this.lt(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 小于等于
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C le(String field, Object value) {
    this.wrapper.le(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 小于等于
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C le(Enum<?> field, Object value) {
    return this.le(this.resolveFieldName(field), value);
  }

  /**
   * 小于等于
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C le(String tableAlias, Enum<?> field, Object value) {
    return this.le(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 范围
   *
   * @param field 字段名
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C between(String field, Object leftValue, Object rightValue) {
    this.wrapper.between(this.convertToDbFieldName(field), leftValue, rightValue);
    return childThis;
  }

  /**
   * 范围
   *
   * @param field 字段枚举
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C between(Enum<?> field, Object leftValue, Object rightValue) {
    return this.between(this.resolveFieldName(field), leftValue, rightValue);
  }

  /**
   * 范围
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C between(String tableAlias, Enum<?> field, Object leftValue, Object rightValue) {
    return this.between(this.resolveFieldName(tableAlias, field), leftValue, rightValue);
  }

  /**
   * 不在范围
   *
   * @param field 字段名
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C notBetween(String field, Object leftValue, Object rightValue) {
    this.wrapper.notBetween(this.convertToDbFieldName(field), leftValue, rightValue);
    return childThis;
  }

  /**
   * 不在范围
   *
   * @param field 字段枚举
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C notBetween(Enum<?> field, Object leftValue, Object rightValue) {
    return this.notBetween(this.resolveFieldName(field), leftValue, rightValue);
  }

  /**
   * 不在范围
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param leftValue 左值
   * @param rightValue 右值
   * @return 具体实现
   */
  @Override
  public C notBetween(String tableAlias, Enum<?> field, Object leftValue, Object rightValue) {
    return this.notBetween(this.resolveFieldName(tableAlias, field), leftValue, rightValue);
  }

  /**
   * 模糊匹配
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C like(String field, Object value) {
    this.wrapper.like(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 模糊匹配
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C like(Enum<?> field, Object value) {
    return this.like(this.resolveFieldName(field), value);
  }

  /**
   * 模糊匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C like(String tableAlias, Enum<?> field, Object value) {
    return this.like(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 非模糊匹配
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C notLike(String field, Object value) {
    this.wrapper.notLike(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 非模糊匹配
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C notLike(Enum<?> field, Object value) {
    return this.notLike(this.resolveFieldName(field), value);
  }

  /**
   * 非模糊匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C notLike(String tableAlias, Enum<?> field, Object value) {
    return this.notLike(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 左模糊匹配
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeLeft(String field, Object value) {
    this.wrapper.likeLeft(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 左模糊匹配
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeLeft(Enum<?> field, Object value) {
    return this.likeLeft(this.resolveFieldName(field), value);
  }

  /**
   * 左模糊匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeLeft(String tableAlias, Enum<?> field, Object value) {
    return this.likeLeft(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 右模糊匹配
   *
   * @param field 字段名
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeRight(String field, Object value) {
    this.wrapper.likeRight(this.convertToDbFieldName(field), value);
    return childThis;
  }

  /**
   * 右模糊匹配
   *
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeRight(Enum<?> field, Object value) {
    return this.likeRight(this.resolveFieldName(field), value);
  }

  /**
   * 右模糊匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 值
   * @return 具体实现
   */
  @Override
  public C likeRight(String tableAlias, Enum<?> field, Object value) {
    return this.likeRight(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 范围匹配
   *
   * @param field 字段名
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C in(String field, Collection<?> values) {
    this.wrapper.in(this.convertToDbFieldName(field), values);
    return childThis;
  }

  /**
   * 范围匹配
   *
   * @param field 字段枚举
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C in(Enum<?> field, Collection<?> values) {
    return this.in(this.resolveFieldName(field), values);
  }

  /**
   * 范围匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C in(String tableAlias, Enum<?> field, Collection<?> values) {
    return this.in(this.resolveFieldName(tableAlias, field), values);
  }

  /**
   * 范围匹配
   *
   * @param field 字段名
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C in(String field, Object... values) {
    this.wrapper.in(this.convertToDbFieldName(field), values);
    return childThis;
  }

  /**
   * 范围匹配
   *
   * @param field 字段枚举
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C in(Enum<?> field, Object... values) {
    return this.in(this.resolveFieldName(field), values);
  }

  /**
   * 范围匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C in(String tableAlias, Enum<?> field, Object... values) {
    return this.in(this.resolveFieldName(tableAlias, field), values);
  }

  /**
   * 范围不匹配
   *
   * @param field 字段名
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C notIn(String field, Collection<?> values) {
    this.wrapper.notIn(this.convertToDbFieldName(field), values);
    return childThis;
  }

  /**
   * 范围不匹配
   *
   * @param field 字段枚举
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C notIn(Enum<?> field, Collection<?> values) {
    return this.notIn(this.resolveFieldName(field), values);
  }

  /**
   * 范围不匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param values 值集合
   * @return 具体实现
   */
  @Override
  public C notIn(String tableAlias, Enum<?> field, Collection<?> values) {
    return this.notIn(this.resolveFieldName(tableAlias, field), values);
  }

  /**
   * 范围不匹配
   *
   * @param field 字段名
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C notIn(String field, Object... values) {
    this.wrapper.notIn(this.convertToDbFieldName(field), values);
    return childThis;
  }

  /**
   * 范围不匹配
   *
   * @param field 字段枚举
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C notIn(Enum<?> field, Object... values) {
    return this.notIn(this.resolveFieldName(field), values);
  }

  /**
   * 范围不匹配
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param values 值数组
   * @return 具体实现
   */
  @Override
  public C notIn(String tableAlias, Enum<?> field, Object... values) {
    return this.notIn(this.resolveFieldName(tableAlias, field), values);
  }

  /**
   * 字段 = ( sql语句 )
   *
   * <p>例1: eqSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C eqSql(String field, String sqlValue) {

    this.wrapper.apply(StrFormatter.format("{}=({})", this.convertToDbFieldName(field), sqlValue));

    return this.childThis;
  }

  /**
   * 字段 = ( sql语句 )
   *
   * <p>例1: eqSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C eqSql(Enum<?> field, String sqlValue) {
    return this.eqSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 = ( sql语句 )
   *
   * <p>例1: eqSql("id", "select id from table where name = 'JunJun'")
   *
   * @param tableAlias 字段表别名
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C eqSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.eqSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 &gt; ( sql语句 )
   *
   * <p>例1: gtSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C gtSql(String field, String sqlValue) {
    this.wrapper.gtSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 &gt; ( sql语句 )
   *
   * <p>例1: gtSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C gtSql(Enum<?> field, String sqlValue) {
    return this.gtSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 &gt; ( sql语句 )
   *
   * <p>例1: gtSql("id", "select id from table where name = 'JunJun'")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C gtSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.gtSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 >= ( sql语句 )
   *
   * <p>例1: geSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: geSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C geSql(String field, String sqlValue) {
    this.wrapper.geSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 >= ( sql语句 )
   *
   * <p>例1: geSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: geSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C geSql(Enum<?> field, String sqlValue) {
    return this.geSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 >= ( sql语句 )
   *
   * <p>例1: geSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: geSql("id", "select id from table where name = 'JunJun'")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C geSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.geSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 &lt; ( sql语句 )
   *
   * <p>例1: ltSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: ltSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C ltSql(String field, String sqlValue) {
    this.wrapper.ltSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 &lt; ( sql语句 )
   *
   * <p>例1: ltSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: ltSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C ltSql(Enum<?> field, String sqlValue) {
    return this.ltSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 &lt; ( sql语句 )
   *
   * <p>例1: ltSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: ltSql("id", "select id from table where name = 'JunJun'")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C ltSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.ltSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 <= ( sql语句 )
   *
   * <p>例1: leSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: leSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段名
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C leSql(String field, String sqlValue) {
    this.wrapper.leSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 <= ( sql语句 )
   *
   * <p>例1: leSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: leSql("id", "select id from table where name = 'JunJun'")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C leSql(Enum<?> field, String sqlValue) {
    return this.leSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 <= ( sql语句 )
   *
   * <p>例1: leSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例1: leSql("id", "select id from table where name = 'JunJun'")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return 具体实现
   */
  @Override
  public C leSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.leSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 in 方法 !!
   *
   * <p>例1: inSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: inSql("id", "select id from table where id &lt; 3")
   *
   * @param field 字段
   * @param sqlValue sql语句
   * @return children
   */
  @Override
  public C inSql(String field, String sqlValue) {
    this.wrapper.inSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 in 方法 !!
   *
   * <p>例1: inSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: inSql("id", "select id from table where id &lt; 3")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return children
   */
  @Override
  public C inSql(Enum<?> field, String sqlValue) {
    return this.inSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 in 方法 !!
   *
   * <p>例1: inSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: inSql("id", "select id from table where id &lt; 3")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句
   * @return children
   */
  @Override
  public C inSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.inSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 NOT IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 not in 方法 !!
   *
   * <p>例1: notInSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: notInSql("id", "select id from table where id &lt; 3")
   *
   * @param field 字段
   * @param sqlValue sql语句 ---&gt; 1,2,3,4,5,6 或者 select id from table where id &lt; 3
   * @return children
   */
  @Override
  public C notInSql(String field, String sqlValue) {
    this.wrapper.notInSql(this.convertToDbFieldName(field), sqlValue);
    return this.childThis;
  }

  /**
   * 字段 NOT IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 not in 方法 !!
   *
   * <p>例1: notInSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: notInSql("id", "select id from table where id &lt; 3")
   *
   * @param field 字段枚举
   * @param sqlValue sql语句 ---&gt; 1,2,3,4,5,6 或者 select id from table where id &lt; 3
   * @return children
   */
  @Override
  public C notInSql(Enum<?> field, String sqlValue) {
    return this.notInSql(this.resolveFieldName(field), sqlValue);
  }

  /**
   * 字段 NOT IN ( sql语句 )
   *
   * <p>!! sql 注入方式的 not in 方法 !!
   *
   * <p>例1: notInSql("id", "1, 2, 3, 4, 5, 6")
   *
   * <p>例2: notInSql("id", "select id from table where id &lt; 3")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param sqlValue sql语句 ---&gt; 1,2,3,4,5,6 或者 select id from table where id &lt; 3
   * @return children
   */
  @Override
  public C notInSql(String tableAlias, Enum<?> field, String sqlValue) {
    return this.notInSql(this.resolveFieldName(tableAlias, field), sqlValue);
  }

  /**
   * 字段 IS NULL
   *
   * <p>例: isNull("name")
   *
   * @param field 字段
   * @return children
   */
  @Override
  public C isNull(String field) {
    this.wrapper.isNull(this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 字段 IS NULL
   *
   * <p>例: isNull("name")
   *
   * @param field 字段枚举
   * @return children
   */
  @Override
  public C isNull(Enum<?> field) {
    return this.isNull(this.resolveFieldName(field));
  }

  /**
   * 字段 IS NULL
   *
   * <p>例: isNull("name")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @return children
   */
  @Override
  public C isNull(String tableAlias, Enum<?> field) {
    return this.isNull(this.resolveFieldName(tableAlias, field));
  }

  /**
   * 字段 IS NOT NULL
   *
   * <p>例: isNotNull("name")
   *
   * @param field 字段
   * @return children
   */
  @Override
  public C isNotNull(String field) {
    this.wrapper.isNotNull(this.convertToDbFieldName(field));
    return this.childThis;
  }

  /**
   * 字段 IS NOT NULL
   *
   * <p>例: isNotNull("name")
   *
   * @param field 字段枚举
   * @return children
   */
  @Override
  public C isNotNull(Enum<?> field) {
    return this.isNotNull(this.resolveFieldName(field));
  }

  /**
   * 字段 IS NOT NULL
   *
   * <p>例: isNotNull("name")
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @return children
   */
  @Override
  public C isNotNull(String tableAlias, Enum<?> field) {
    return this.isNotNull(this.resolveFieldName(tableAlias, field));
  }

  /**
   * 拼接 sql
   *
   * <p>!! 会有 sql 注入风险 !!
   *
   * <p>例1: expression("id = 1")
   *
   * <p>例2: expression("date_format(dateColumn,'%Y-%m-%d') = '2008-08-08'")
   *
   * <p>例3: expression("date_format(dateColumn,'%Y-%m-%d') = {0}", LocalDate.now())
   *
   * @param sqlValue sql语句
   * @param values 数据数组
   * @return children
   */
  @Override
  public C expression(String sqlValue, Object... values) {
    this.getJoin().apply(sqlValue, values);
    return this.childThis;
  }

  /**
   * 拼接 EXISTS ( sql语句 )
   *
   * <p>!! sql 注入方法 !!
   *
   * <p>例: exists("select id from table where age = 1")
   *
   * @param sqlValue sql语句
   * @param values 数据数组
   * @return children
   */
  @Override
  public C exists(String sqlValue, Object... values) {
    this.getJoin().exists(sqlValue, values);
    return this.childThis;
  }

  /**
   * 拼接 NOT EXISTS ( sql语句 )
   *
   * <p>!! sql 注入方法 !!
   *
   * <p>例: notExists("select id from table where age = 1")
   *
   * @param sqlValue sql语句
   * @param values 数据数组
   * @return children
   */
  @Override
  public C notExists(String sqlValue, Object... values) {
    this.getJoin().notExists(sqlValue, values);
    return this.childThis;
  }

  /**
   * 应用action消费器
   *
   * @param actionConsumer action消费器
   * @return 具体实现
   */
  @Override
  public C apply(Consumer<C> actionConsumer) {
    Assert.notNull(actionConsumer, "Action消费器不能为空");
    actionConsumer.accept(this.childThis);
    return this.childThis;
  }

  /**
   * 获取操作Action实例
   *
   * @param wrapper 条件Wrapper对象
   * @return Action实例
   */
  protected abstract C instance(Wrapper<E> wrapper);

  /**
   * 拼接 OR
   *
   * @return children
   */
  @Override
  public C or() {
    this.getJoin().or();
    return this.childThis;
  }

  /**
   * OR 嵌套
   *
   * <p>例: or(i -&gt; i.eq("name", "李白").ne("status", "活着"))
   *
   * @param consumer 消费函数
   * @return children
   */
  @Override
  public C or(Consumer<C> consumer) {
    this.getNested()
        .or(
            wp -> {
              C child = this.instance(wp);
              consumer.accept(child);
            });
    return this.childThis;
  }

  /**
   * AND 嵌套
   *
   * <p>例: and(i -&gt; i.eq("name", "李白").ne("status", "活着"))
   *
   * @param consumer 消费函数
   * @return children
   */
  @Override
  public C and(Consumer<C> consumer) {
    this.getNested()
        .and(
            wp -> {
              C child = this.instance(wp);
              consumer.accept(child);
            });
    return this.childThis;
  }

  /**
   * 正常嵌套 不带 AND 或者 OR
   *
   * <p>例: nested(i -&gt; i.eq("name", "李白").ne("status", "活着"))
   *
   * @param consumer 消费函数
   * @return children
   */
  @Override
  public C nested(Consumer<C> consumer) {
    this.getNested()
        .nested(
            wp -> {
              C child = this.instance(wp);
              consumer.accept(child);
            });
    return this.childThis;
  }
}
