package org.elsfs.tool.sql.mybatisplus.mybatis;

import com.baomidou.mybatisplus.core.conditions.query.Query;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.core.metadata.TableInfoHelper;
import java.util.Collection;
import java.util.Collections;
import java.util.function.Consumer;
import org.elsfs.tool.core.text.StrUtils;
import org.elsfs.tool.sql.abs.AbstractCondition;
import org.elsfs.tool.sql.abs.CompositeSqlFragment;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.common.FieldEnumResolverSupport;
import org.elsfs.tool.sql.condition.ConditionItem;
import org.elsfs.tool.sql.interfaces.SqlParameterManager;
import org.elsfs.tool.sql.interfaces.TableAliasManager;
import org.elsfs.tool.sql.mybatisplus.extension.JoinQueryWrapper;
import org.elsfs.tool.sql.singular.facade.ConditionFields;
import org.elsfs.tool.sql.singular.facade.EntityRelationDescriptor;
import org.elsfs.tool.sql.singular.facade.QueryAction;
import org.elsfs.tool.sql.utils.SqlPool;
import org.elsfs.tool.sql.utils.SqlUtils;

/**
 * MybatisPlus实现
 *
 * @param <E> 实体类型
 * @param <R> 返回类型
 * @author zeng
 * @since 0.0.4
 */
public abstract class AbstractMybatisPlusQueryAction<E, R>
    extends AbstractMybatisPlusQueryOperations<QueryAction<R>, E> implements QueryAction<R> {

  /** 查询Wrapper */
  protected final JoinQueryWrapper<E> queryWrapper;

  /** 表别名管理器 */
  protected final TableAliasManager tableAliasManager;

  protected AbstractMybatisPlusQueryAction(
      JoinQueryWrapper<E> queryWrapper, TableAliasManager tableAliasManager) {
    super(queryWrapper);
    this.queryWrapper = queryWrapper;
    this.tableAliasManager = tableAliasManager;
  }

  /**
   * 解析字段枚举名称
   *
   * @param field 字段枚举
   * @return 字段名称
   */
  @Override
  public String resolveFieldName(Enum<?> field) {
    return FieldEnumResolverSupport.resolveFieldName(this.tableAliasManager, field);
  }

  /**
   * 解析字段枚举名称
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @return 字段名称
   */
  @Override
  public String resolveFieldName(String tableAlias, Enum<?> field) {
    return tableAlias + SqlPool.DOT + field.name();
  }

  /**
   * 获取Query查询对象
   *
   * @return 查询对象
   */
  @Override
  protected Query<?, E, String> query() {
    return this.queryWrapper;
  }

  /**
   * 获取条件字段 注意：考虑性能问题，此方法不可重复调用，请尽量调用一次，保存中间结果变量
   *
   * @return 条件字段
   */
  @Override
  public ConditionFields getConditionFields() {
    String conditionSql = this.queryWrapper.getTargetSql();
    if (StrUtils.isBlank(conditionSql)) {
      return new ConditionFields(Collections.emptySet(), this.tableAliasManager);
    }
    return new ConditionFields(
        SqlUtils.parseConditionSqlColumnNames(conditionSql), this.tableAliasManager);
  }

  /**
   * 清空默认选择字段
   *
   * @return 具体实现
   */
  @Override
  public QueryAction<R> clearDefaultSelectedFields() {
    this.queryWrapper.clearDefaultSelectedFields();
    return this;
  }

  /**
   * 快速左连接关联实体
   *
   * @param relationDescriptor 关联描述符对象
   * @return this
   */
  @Override
  public QueryAction<R> associate(EntityRelationDescriptor relationDescriptor) {
    TableInfo tableInfo = TableInfoHelper.getTableInfo(relationDescriptor.getEntityClass());

    return this.leftJoin(relationDescriptor.getEntityClass())
        .select(relationDescriptor.getSelectionFields().toArray(String[]::new))
        .on()
        .eq(
            JoinQueryWrapper.TABLE_ALIAS_PREFIX + relationDescriptor.getForeignKeyField(),
            this.tableAliasManager.fetchTableAlias(tableInfo.getTableName())
                + SqlPool.DOT
                + (StrUtils.isBlank(relationDescriptor.getJoinKeyField())
                    ? tableInfo.getKeyColumn()
                    : relationDescriptor.getJoinKeyField()))
        .end();
  }

  /**
   * 构建SQL片段
   *
   * @return SQL片段
   */
  @Override
  public String buildSqlFragment() {
    return this.queryWrapper.buildSqlFragment();
  }

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(Class<?> entityClass) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.join(entityClass));
  }

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(
      Class<?> entityClass, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.join(entityClass, tableAlias, registerAlias));
  }

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(Class<?> entityClass, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.join(entityClass, tableAlias));
  }

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(String table, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.join(table, tableAlias));
  }

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(String table) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.join(table));
  }

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(String table, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.join(table, tableAlias, registerAlias));
  }

  /**
   * 内连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> join(
      Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.join(sqlBuilderConsumer, tableAlias));
  }

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(Class<?> entityClass, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.leftJoin(entityClass, tableAlias));
  }

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(Class<?> entityClass) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.leftJoin(entityClass));
  }

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(String table) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.leftJoin(table));
  }

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(
      Class<?> entityClass, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.leftJoin(entityClass, tableAlias, registerAlias));
  }

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(String table, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.leftJoin(table, tableAlias));
  }

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(
      String table, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.leftJoin(table, tableAlias, registerAlias));
  }

  /**
   * 左连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> leftJoin(
      Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.leftJoin(sqlBuilderConsumer, tableAlias));
  }

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(Class<?> entityClass) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.rightJoin(entityClass));
  }

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(Class<?> entityClass, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.rightJoin(entityClass, tableAlias));
  }

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(
      Class<?> entityClass, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.rightJoin(entityClass, tableAlias, registerAlias));
  }

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(String table) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.rightJoin(table));
  }

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(String table, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(this.queryWrapper.rightJoin(table, tableAlias));
  }

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(
      String table, String tableAlias, boolean registerAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.rightJoin(table, tableAlias, registerAlias));
  }

  /**
   * 右连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  @Override
  public ActionSelectJoinBuilder<R> rightJoin(
      Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias) {
    return new ActionSelectJoinBuilderImpl(
        this.queryWrapper.rightJoin(sqlBuilderConsumer, tableAlias));
  }

  /** 自定义Join配置实现 */
  public class ActionSelectJoinBuilderImpl extends CompositeSqlFragment
      implements ActionSelectJoinBuilder<R> {

    /** Join构建器代理 */
    private final JoinQueryWrapper<E>.WrapperSelectJoinBuilderImpl delegate;

    public ActionSelectJoinBuilderImpl(JoinQueryWrapper<E>.WrapperSelectJoinBuilderImpl delegate) {
      this.delegate = delegate;
    }

    /**
     * 返回上一级对象
     *
     * @return 上级对象
     */
    @Override
    public QueryAction<R> end() {
      return AbstractMybatisPlusQueryAction.this;
    }

    /**
     * 连接条件设置
     *
     * @return 连接条件构建器
     */
    @Override
    public ActionJoinConditionBuilder<R> on() {
      return new ActionJoinConditionConfigurerImpl(this.delegate.on());
    }

    /**
     * 选择关联SQL字段
     *
     * @param fields SQL字段枚举
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> select(Enum<?>... fields) {
      this.delegate.select(fields);
      return this;
    }

    /**
     * 选择关联SQL字段
     *
     * @param fields SQL字段
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> select(String... fields) {
      this.delegate.select(fields);
      return this;
    }

    /**
     * 选择关联SQL字段
     *
     * @param fields SQL字段
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> select(Collection<String> fields) {
      this.delegate.select(fields);
      return this;
    }

    /**
     * 选择关联SQL字段表达式
     *
     * @param expressions SQL字段表达式
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectExpression(String... expressions) {
      this.delegate.selectExpression(expressions);
      return this;
    }

    /**
     * 选择关联SQL字段表达式
     *
     * @param expressions SQL字段表达式
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectExpression(Collection<String> expressions) {
      this.delegate.selectExpression(expressions);
      return this;
    }

    /**
     * 选择SQL字段
     *
     * @param field SQL字段
     * @param fieldAlias 字段别名
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectAs(String field, String fieldAlias) {
      this.delegate.selectAs(field, fieldAlias);
      return this;
    }

    /**
     * 选择SQL字段
     *
     * @param field SQL字段枚举
     * @param fieldAlias 字段别名
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectAs(Enum<?> field, String fieldAlias) {
      this.delegate.selectAs(field, fieldAlias);
      return this;
    }

    /**
     * 选择SQL字段
     *
     * @param field SQL字段枚举
     * @param fieldAlias 字段别名枚举
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectAs(Enum<?> field, Enum<?> fieldAlias) {
      this.delegate.selectAs(field, fieldAlias);
      return this;
    }

    /**
     * 选择子查询结果
     *
     * @param sqlBuilderConsumer SQL构建器消费器
     * @param fieldAlias 字段别名
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectAs(
        Consumer<StandardSelectSql> sqlBuilderConsumer, String fieldAlias) {
      this.delegate.selectAs(sqlBuilderConsumer, fieldAlias);
      return this;
    }

    /**
     * 选择子查询结果
     *
     * @param sqlBuilderConsumer SQL构建器消费器
     * @param fieldAlias 字段别名枚举
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectAs(
        Consumer<StandardSelectSql> sqlBuilderConsumer, Enum<?> fieldAlias) {
      this.delegate.selectAs(sqlBuilderConsumer, fieldAlias);
      return this;
    }

    /**
     * 选择关联SQL字段
     *
     * @param fields SQL字段枚举
     * @return 具体实现
     */
    @Override
    public ActionSelectJoinBuilder<R> selectEnum(Collection<Enum<?>> fields) {
      this.delegate.selectEnum(fields);
      return this;
    }

    /** 自定义Join条件配置器 */
    public class ActionJoinConditionConfigurerImpl
        extends AbstractCondition<ActionJoinConditionBuilder<R>>
        implements ActionJoinConditionBuilder<R> {

      /** Join条件构建器代理 */
      private final JoinQueryWrapper<E>.WrapperSelectJoinBuilderImpl.WrapperJoinConditionBuilderImpl
          delegate;

      public ActionJoinConditionConfigurerImpl(
          JoinQueryWrapper<E>.WrapperSelectJoinBuilderImpl.WrapperJoinConditionBuilderImpl
              delegate) {
        this.delegate = delegate;
      }

      /**
       * 添加一个条件
       *
       * @param conditionItem 条件
       */
      @Override
      protected void addConditionItem(ConditionItem conditionItem) {
        this.delegate.addConditionItem(conditionItem);
      }

      /**
       * 返回上一级对象
       *
       * @return 上级对象
       */
      @Override
      public QueryAction<R> end() {
        return ActionSelectJoinBuilderImpl.this.end();
      }

      /**
       * 获取SQL参数管理器
       *
       * @return SQL参数管理器
       */
      @Override
      public SqlParameterManager getSqlParameterManager() {
        return this.delegate.getSqlParameterManager();
      }

      /**
       * 获取表别名管理器
       *
       * @return 表名别管理器
       */
      @Override
      public TableAliasManager getTableAliasManager() {
        return this.delegate.getTableAliasManager();
      }
    }
  }
}
