package org.elsfs.tool.sql.mybatisplus.mybatis;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import java.util.List;
import org.elsfs.tool.sql.common.SimpleTableAliasManager;
import org.elsfs.tool.sql.interfaces.TableAliasManager;
import org.elsfs.tool.sql.mybatisplus.extension.JoinQueryWrapper;
import org.elsfs.tool.sql.mybatisplus.mapper.ElsfsMapper;
import org.elsfs.tool.core.page.IPage;
import org.elsfs.tool.sql.singular.facade.DbActionFactory;
import org.elsfs.tool.sql.singular.facade.DeletionAction;
import org.elsfs.tool.sql.singular.facade.QueryAction;
import org.elsfs.tool.sql.singular.facade.UpdateAction;

/**
 * MybatisPlus工厂实现
 *
 * @param <E> 实体类型
 * @author zeng
 * @since 0.0.4
 */
public class MybatisPlusDbActionFactory<E> implements DbActionFactory<E> {

  /** 基础Mapper */
  private final ElsfsMapper<E> vefMapper;

  /** 实体类 */
  private final Class<E> entityClass;

  public MybatisPlusDbActionFactory(ElsfsMapper<E> vefMapper, Class<E> entityClass) {
    this.vefMapper = vefMapper;
    this.entityClass = entityClass;
  }

  /**
   * 创建一个单记录查询动作
   *
   * @return 查询动作对象
   */
  @Override
  public QueryAction<E> queryOne() {
    TableAliasManager tableAliasManager = new SimpleTableAliasManager();

    return new SingleMybatisPlusQueryAction<>(
        new JoinQueryWrapper<>(this.entityClass, tableAliasManager),
        this.vefMapper,
        tableAliasManager);
  }

  /**
   * 创建一个多记录查询动作
   *
   * @return 查询动作对象
   */
  @Override
  public QueryAction<List<E>> queryList() {
    TableAliasManager tableAliasManager = new SimpleTableAliasManager();

    return new ListMybatisPlusQueryAction<>(
        new JoinQueryWrapper<>(this.entityClass, tableAliasManager),
        this.vefMapper,
        tableAliasManager);
  }

  /**
   * 创建一个分页查询动作
   *
   * @return 分页查询动作对象
   */
  @Override
  public QueryAction<IPage<E>> queryPage() {
    TableAliasManager tableAliasManager = new SimpleTableAliasManager();

    return new PageMybatisPlusQueryAction<>(
        new JoinQueryWrapper<>(this.entityClass, tableAliasManager),
        this.vefMapper,
        tableAliasManager);
  }

  /**
   * 创建一个统计查询动作
   *
   * @return 查询动作对象
   */
  @Override
  public QueryAction<Long> count() {
    TableAliasManager tableAliasManager = new SimpleTableAliasManager();

    return new CountMybatisPlusQueryAction<>(
        new JoinQueryWrapper<>(this.entityClass, tableAliasManager),
        this.vefMapper,
        tableAliasManager);
  }

  /**
   * 创建一个存在查询动作
   *
   * @return 查询动作对象
   */
  @Override
  public QueryAction<Boolean> exists() {
    TableAliasManager tableAliasManager = new SimpleTableAliasManager();

    return new ExistsMybatisPlusQueryAction<>(
        new JoinQueryWrapper<>(this.entityClass, tableAliasManager),
        this.vefMapper,
        tableAliasManager);
  }

  /**
   * 创建一个更新动作
   *
   * @return 更新动作对象
   */
  @Override
  public UpdateAction<Long> update() {
    return new MybatisPlusUpdateAction<>(Wrappers.update(), this.vefMapper);
  }

  /**
   * 创建一个删除动作
   *
   * @return 删除动作对象
   */
  @Override
  public DeletionAction<Long> delete() {
    return new MybatisPlusDeletionAction<>(Wrappers.query(), this.vefMapper);
  }
}
