package org.elsfs.tool.sql.interfaces.join;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.resolve.ResolveFieldName;
import org.elsfs.tool.sql.utils.CastUtils;

/**
 * in
 *
 * @author zeng
 * @since 0.0.4
 */
public interface JoinIn<Children> extends ResolveFieldName {

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  default Children in(String leftField, String... rightFields) {
    return in(true, leftField, rightFields);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  Children in(boolean condition, String leftField, String... rightFields);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @return 具体实现
   */
  default Children in(Enum<?> leftField, Enum<?>... rightFields) {
    return this.in(
        this.resolveFieldName(leftField),
        Arrays.stream(rightFields).map(this::resolveFieldName).toArray(String[]::new));
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  Children in(String leftField, Collection<String> rightFields);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @return 具体实现
   */
  default Children in(Enum<?> leftField, Collection<Enum<?>> rightFields) {
    return this.in(
        this.resolveFieldName(leftField),
        rightFields.stream().map(this::resolveFieldName).collect(Collectors.toSet()));
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  Children in(String leftField, Collection<String> rightFields, Collection<?> rightValues);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children in(
      Enum<?> leftField, Collection<Enum<?>> rightFields, Collection<?> rightValues) {
    return this.in(
        this.resolveFieldName(leftField),
        rightFields.stream().map(this::resolveFieldName).collect(Collectors.toSet()),
        rightValues);
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  Children in(String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  default Children in(Enum<?> leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    return this.in(this.resolveFieldName(leftField), sqlBuilderConsumer);
  }

  /**
   * 范围条件
   *
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  default Children in(
      String leftTableAlias, Enum<?> leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    return this.in(this.resolveFieldName(leftTableAlias, leftField), sqlBuilderConsumer);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @return 具体实现
   */
  default Children in(boolean condition, Enum<?> leftField, Enum<?>... rightFields) {
    if (condition) {
      return this.in(leftField, rightFields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  default Children in(boolean condition, String leftField, Collection<String> rightFields) {
    if (condition) {
      return this.in(leftField, rightFields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @return 具体实现
   */
  default Children in(boolean condition, Enum<?> leftField, Collection<Enum<?>> rightFields) {
    if (condition) {
      return this.in(leftField, rightFields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children in(
      boolean condition,
      String leftField,
      Collection<String> rightFields,
      Collection<?> rightValues) {
    if (condition) {
      return this.in(leftField, rightFields, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param rightFields 范围字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children in(
      boolean condition,
      Enum<?> leftField,
      Collection<Enum<?>> rightFields,
      Collection<?> rightValues) {
    if (condition) {
      return this.in(leftField, rightFields, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  default Children in(
      boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      return this.in(leftField, sqlBuilderConsumer);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  default Children in(
      boolean condition, Enum<?> leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      return this.in(leftField, sqlBuilderConsumer);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  default Children in(
      boolean condition,
      String leftTableAlias,
      Enum<?> leftField,
      Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      return this.in(leftTableAlias, leftField, sqlBuilderConsumer);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  Children inValue(String leftField, Object... rightValues);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(Enum<?> leftField, Object... rightValues) {
    return this.inValue(this.resolveFieldName(leftField), rightValues);
  }

  /**
   * 范围条件
   *
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(String leftTableAlias, Enum<?> leftField, Object... rightValues) {
    return this.inValue(this.resolveFieldName(leftTableAlias, leftField), rightValues);
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  Children inValue(String leftField, Collection<?> rightValues);

  /**
   * 范围条件
   *
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(Enum<?> leftField, Collection<?> rightValues) {
    return this.inValue(this.resolveFieldName(leftField), rightValues);
  }

  /**
   * 范围条件
   *
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(String leftTableAlias, Enum<?> leftField, Collection<?> rightValues) {
    return this.inValue(this.resolveFieldName(leftTableAlias, leftField), rightValues);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(boolean condition, String leftField, Object... rightValues) {
    if (condition) {
      return this.inValue(leftField, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(boolean condition, Enum<?> leftField, Object... rightValues) {
    if (condition) {
      return this.inValue(leftField, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(
      boolean condition, String leftTableAlias, Enum<?> leftField, Object... rightValues) {
    if (condition) {
      return this.inValue(leftTableAlias, leftField, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(boolean condition, String leftField, Collection<?> rightValues) {
    if (condition) {
      return this.inValue(leftField, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(boolean condition, Enum<?> leftField, Collection<?> rightValues) {
    if (condition) {
      return this.inValue(leftField, rightValues);
    }

    return CastUtils.cast(this);
  }

  /**
   * 范围条件
   *
   * @param condition 条件
   * @param leftTableAlias 左边字段表别名
   * @param leftField 左边字段枚举
   * @param rightValues 范围值
   * @return 具体实现
   */
  default Children inValue(
      boolean condition, String leftTableAlias, Enum<?> leftField, Collection<?> rightValues) {
    if (condition) {
      return this.inValue(leftTableAlias, leftField, rightValues);
    }

    return CastUtils.cast(this);
  }
}
