package org.elsfs.tool.sql.select;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import org.elsfs.tool.core.text.NamingCase;
import org.elsfs.tool.core.text.StrUtils;
import org.elsfs.tool.sql.exception.SqlBuilderException;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 标准选择项实现
 *
 * @author zeng
 * @version 0.0.4
 */
public class StandardSelectItem implements SelectItem {

  /** 字段别名分割正则模式 */
  private static final Pattern FIELD_ALIAS_SPLITTING_PATTERN =
      Pattern.compile("( +as +(?=\\w+$))|( +(?=\\w+$))", Pattern.CASE_INSENSITIVE);

  /** 字段所属表别名 */
  private final String tableAlias;

  /** 字段名称 */
  private final String field;

  /** 字段名别 */
  private final String fieldAlias;

  /**
   * 构造函数
   *
   * @param tableAlias 字段所属表别名
   * @param field 字段名称
   * @param fieldAlias 字段名别
   */
  public StandardSelectItem(String tableAlias, String field, String fieldAlias) {
    this.tableAlias = tableAlias;
    this.field = field;
    this.fieldAlias = fieldAlias;
  }

  public String getTableAlias() {
    return tableAlias;
  }

  public String getField() {
    return field;
  }

  public String getFieldAlias() {
    return fieldAlias;
  }

  /**
   * 构建SQL片段
   *
   * @param field 字段名称
   */
  public StandardSelectItem(String field) {
    if (StrUtils.isBlank(field)) {
      throw new SqlBuilderException("选择字段不能为空");
    }

    String alias = null;
    if (field.contains(SqlPool.DOT)) {
      List<String> parts = Arrays.stream(field.split(SqlPool.DOT)).toList();

      if (parts.size() == 2) {
        alias = parts.get(0);
        field = parts.get(1);
      }
    }

    String[] tokens = FIELD_ALIAS_SPLITTING_PATTERN.split(field);
    if (tokens.length == 0) {
      throw new SqlBuilderException("选择字段不能为空");
    }

    if (tokens.length == 1) {
      this.tableAlias = alias;
      this.field = tokens[0];
      this.fieldAlias = null;
      return;
    }

    this.tableAlias = alias;
    this.field = tokens[0];
    this.fieldAlias = tokens[1];
  }

  /**
   * 构建SQL片段
   *
   * @return SQL片段
   */
  @Override
  public String buildSqlFragment() {
    StringBuilder stringBuilder = new StringBuilder();
    if (StrUtils.isNotBlank(this.tableAlias)) {
      stringBuilder.append(this.tableAlias).append(SqlPool.DOT);
    }
    if (StrUtils.isBlank(this.fieldAlias)) {
      stringBuilder.append(NamingCase.toUnderlineCase(this.field));
    } else {
      stringBuilder
          .append(NamingCase.toUnderlineCase(this.field))
          .append(SqlPool.AS_KEYWORD_WITH_SPACE)
          .append(NamingCase.toUnderlineCase(this.fieldAlias));
    }

    return stringBuilder.toString();
  }
}
