package org.elsfs.tool.sql.utils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.expression.CastExpression;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.ExpressionVisitorAdapter;
import net.sf.jsqlparser.expression.Function;
import net.sf.jsqlparser.expression.NotExpression;
import net.sf.jsqlparser.expression.Parenthesis;
import net.sf.jsqlparser.expression.SignedExpression;
import net.sf.jsqlparser.expression.operators.conditional.AndExpression;
import net.sf.jsqlparser.expression.operators.conditional.OrExpression;
import net.sf.jsqlparser.expression.operators.conditional.XorExpression;
import net.sf.jsqlparser.expression.operators.relational.Between;
import net.sf.jsqlparser.expression.operators.relational.EqualsTo;
import net.sf.jsqlparser.expression.operators.relational.ExpressionList;
import net.sf.jsqlparser.expression.operators.relational.GreaterThan;
import net.sf.jsqlparser.expression.operators.relational.GreaterThanEquals;
import net.sf.jsqlparser.expression.operators.relational.InExpression;
import net.sf.jsqlparser.expression.operators.relational.IsNullExpression;
import net.sf.jsqlparser.expression.operators.relational.LikeExpression;
import net.sf.jsqlparser.expression.operators.relational.MinorThan;
import net.sf.jsqlparser.expression.operators.relational.MinorThanEquals;
import net.sf.jsqlparser.expression.operators.relational.NotEqualsTo;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.schema.Column;
import org.elsfs.tool.core.text.ArrayUtils;
import org.elsfs.tool.core.text.StrFormatter;
import org.elsfs.tool.core.text.StrUtils;
import org.elsfs.tool.core.util.DateTimeUtils;
import org.elsfs.tool.sql.exception.SqlBuilderException;

/**
 * SQL工具类
 *
 * @author zeng
 * @version 0.0.4
 */
public final class SqlUtils {

  /**
   * 处理SQL参数值
   *
   * @param value 参数值
   * @return 处理后的参数
   */
  public static String handleSqlParameterValue(Object value) {
    if (value instanceof CharSequence) {
      return SqlPool.SINGLE_QUOTE + value + SqlPool.SINGLE_QUOTE;
    } else if (value instanceof Enum<?> enumeration) {
      return SqlPool.SINGLE_QUOTE + enumeration.name() + SqlPool.SINGLE_QUOTE;
    } else if (value instanceof Date date) {
      return SqlPool.SINGLE_QUOTE + DateTimeUtils.toLocalDateTime(date) + SqlPool.SINGLE_QUOTE;
    } else if (value instanceof LocalDateTime
        || value instanceof LocalDate
        || value instanceof LocalTime) {
      return SqlPool.SINGLE_QUOTE + value + SqlPool.SINGLE_QUOTE;
    } else if (value == null) {
      return SqlPool.SQL_NULL;
    } else {
      return value.toString();
    }
  }

  /**
   * 处理SQL表达式
   *
   * @param sqlExpression SQL表达式
   * @param args SQL表达式参数
   * @return 处理参数后的SQL表达式
   */
  public static String handleSqlExpression(String sqlExpression, Object... args) {
    if (StrUtils.isBlank(sqlExpression) || ArrayUtils.isEmpty(args)) {
      return sqlExpression;
    }

    Object[] processedArgs = new Object[args.length];
    for (int i = 0; i < args.length; i++) {
      processedArgs[i] = handleSqlParameterValue(args[i]);
    }

    return StrFormatter.format(sqlExpression, processedArgs);
  }

  /**
   * 解析条件SQL列名
   *
   * @param conditionSql 条件SQL
   * @return 条件列名集合
   */
  public static Set<String> parseConditionSqlColumnNames(String conditionSql) {
    return parseConditionSqlColumns(conditionSql).stream()
        .map(Column::getFullyQualifiedName)
        .collect(Collectors.toSet());
  }

  /**
   * 解析条件SQL列
   *
   * @param conditionSql 条件SQL
   * @return 条件列集合
   */
  public static Set<Column> parseConditionSqlColumns(String conditionSql) {
    try {
      Expression expression = CCJSqlParserUtil.parseCondExpression(conditionSql);
      WhereConditionExpressionVisitor expressionVisitor = new WhereConditionExpressionVisitor();
      expression.accept(expressionVisitor);

      return expressionVisitor.columns;
    } catch (JSQLParserException e) {
      throw new SqlBuilderException("解析条件SQL条件列出现异常：" + e.getMessage());
    }
  }

  /** 条件表达式访问器 */
  static class WhereConditionExpressionVisitor extends ExpressionVisitorAdapter {

    /** 条件字段列名 */
    private final Set<Column> columns = new HashSet<>();

    /**
     * 访问函数表达式
     *
     * @param function 函数表达式
     */
    @Override
    public void visit(Function function) {
      ExpressionList parameters = function.getParameters();
      if (parameters == null) {
        return;
      }

      for (Expression expression : parameters.getExpressions()) {
        expression.accept(this);
      }
    }

    /**
     * 方法圆括号表达式
     *
     * @param parenthesis 圆括号表达式
     */
    @Override
    public void visit(Parenthesis parenthesis) {
      parenthesis.getExpression().accept(this);
    }

    /**
     * 访问And表达式
     *
     * @param expr And表达式
     */
    @Override
    public void visit(AndExpression expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问Or表达式
     *
     * @param expr Or表达式
     */
    @Override
    public void visit(OrExpression expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问Xor表达式
     *
     * @param expr Xor表达式
     */
    @Override
    public void visit(XorExpression expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问Between表达式
     *
     * @param expr Between表达式
     */
    @Override
    public void visit(Between expr) {
      expr.getLeftExpression().accept(this);
      expr.getBetweenExpressionStart().accept(this);
      expr.getBetweenExpressionEnd().accept(this);
    }

    /**
     * 访问等于表达式
     *
     * @param expr 等于表达式
     */
    @Override
    public void visit(EqualsTo expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问大于表达式
     *
     * @param expr 大于表达式
     */
    @Override
    public void visit(GreaterThan expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问大于等于表达式
     *
     * @param expr 大于等于表达式
     */
    @Override
    public void visit(GreaterThanEquals expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问In表达式
     *
     * @param expr In表达式
     */
    @Override
    public void visit(InExpression expr) {
      expr.getLeftExpression().accept(this);
      if (expr.getRightItemsList() != null) {
        expr.getRightItemsList().accept(this);
      } else if (expr.getRightExpression() != null) {
        expr.getRightExpression().accept(this);
      }
    }

    /**
     * 访问为空表达式
     *
     * @param expr 为空表达式
     */
    @Override
    public void visit(IsNullExpression expr) {
      expr.getLeftExpression().accept(this);
    }

    /**
     * 访问模糊匹配表达式
     *
     * @param expr 模糊匹配表达式
     */
    @Override
    public void visit(LikeExpression expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问小于表达式
     *
     * @param expr 小于表达式
     */
    @Override
    public void visit(MinorThan expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问小于等于表达式
     *
     * @param expr 小于等于表达式
     */
    @Override
    public void visit(MinorThanEquals expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问不等于表达式
     *
     * @param expr 不等于表达式
     */
    @Override
    public void visit(NotEqualsTo expr) {
      expr.getLeftExpression().accept(this);
      expr.getRightExpression().accept(this);
    }

    /**
     * 访问列表达式
     *
     * @param column 列表达式
     */
    @Override
    public void visit(Column column) {
      this.columns.add(column);
    }

    /**
     * 访问数据类型转换表达式
     *
     * @param expr 数据类型转换表达式
     */
    @Override
    public void visit(CastExpression expr) {
      expr.getLeftExpression().accept(this);
    }

    /**
     * 访问取反表达式
     *
     * @param notExpr 取反表达式
     */
    @Override
    public void visit(NotExpression notExpr) {
      notExpr.getExpression().accept(this);
    }

    /**
     * 访问符号表达式
     *
     * @param expr 符号表达式
     */
    @Override
    public void visit(SignedExpression expr) {
      expr.getExpression().accept(this);
    }
  }
}
