package org.elsfs.tool.sql.interfaces;

import java.util.function.Consumer;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.interfaces.select.Selectable;

/**
 * JOIN连接SQL
 *
 * @param <C> 子类具体类型
 * @param <JB> JOIN构建器类型
 * @param <JCB> JOIN条件构建器类型
 * @author zeng
 * @since 0.0.4
 */
public interface Join<
        C extends Join<C, JB, JCB>,
        JB extends Join.JoinBuilder<JB, JCB, C>,
        JCB extends Join.JoinConditionBuilder<JCB, JB, C>>
    extends SqlFragment {

  /** 主表别名 */
  String TABLE_ALIAS = "t";

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  JB join(Class<?> entityClass);

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB join(Class<?> entityClass, String tableAlias);

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  JB join(String table);

  /**
   * 内连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB join(Class<?> entityClass, String tableAlias, boolean registerAlias);

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB join(String table, String tableAlias);

  /**
   * 内连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB join(String table, String tableAlias, boolean registerAlias);

  /**
   * 内连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB join(Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias);

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  JB rightJoin(Class<?> entityClass);

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB rightJoin(String table, String tableAlias, boolean registerAlias);

  /**
   * 右连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB rightJoin(Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias);

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB rightJoin(Class<?> entityClass, String tableAlias);

  /**
   * 右连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB rightJoin(Class<?> entityClass, String tableAlias, boolean registerAlias);

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  JB rightJoin(String table);

  /**
   * 右连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB rightJoin(String table, String tableAlias);

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @return 连接配置对象
   */
  JB leftJoin(Class<?> entityClass);

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB leftJoin(Class<?> entityClass, String tableAlias);

  /**
   * 左连接一张表
   *
   * @param entityClass 实体类
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB leftJoin(Class<?> entityClass, String tableAlias, boolean registerAlias);

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @return 连接配置对象
   */
  JB leftJoin(String table);

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB leftJoin(String table, String tableAlias);

  /**
   * 左连接一张表
   *
   * @param table 数据库表名
   * @param tableAlias 表别名
   * @param registerAlias 是否注册别名
   * @return 连接配置对象
   */
  JB leftJoin(String table, String tableAlias, boolean registerAlias);

  /**
   * 左连接一个子查询
   *
   * @param sqlBuilderConsumer 子查询构建器
   * @param tableAlias 表别名
   * @return 连接配置对象
   */
  JB leftJoin(Consumer<StandardSelectSql> sqlBuilderConsumer, String tableAlias);

  /**
   * Join构建器
   *
   * @param <C> 具体实现
   * @param <JCB> 条件构建器类型
   * @param <J> 上级对象类型
   */
  interface JoinBuilder<
          C extends JoinBuilder<C, JCB, J>,
          JCB extends JoinConditionBuilder<JCB, C, J>,
          J extends Join<J, C, JCB>>
      extends SqlFragment, Completable<J> {

    /**
     * 连接条件设置
     *
     * @return 连接条件构建器
     */
    JCB on();
  }

  /**
   * 选择Join构建器
   *
   * @param <C> 具体实现
   * @param <JCB> 条件构建器类型
   * @param <J> 上级对象类型
   */
  interface SelectJoinBuilder<
          C extends SelectJoinBuilder<C, JCB, J>,
          JCB extends JoinConditionBuilder<JCB, C, J>,
          J extends Join<J, C, JCB>>
      extends JoinBuilder<C, JCB, J>, Selectable<C> {}

  /**
   * Join条件构建器接口
   *
   * @param <C> 具体实现类型
   * @param <JB> Join构建器对象类型
   * @param <J> 上级对象类型
   */
  interface JoinConditionBuilder<
          C extends JoinConditionBuilder<C, JB, J>,
          JB extends JoinBuilder<JB, C, J>,
          J extends Join<J, JB, C>>
      extends Condition<C>, Completable<J> {}
}
