package org.elsfs.tool.sql.select;

import java.util.regex.Pattern;
import org.elsfs.tool.core.text.NamingCase;
import org.elsfs.tool.core.text.StrUtils;
import org.elsfs.tool.sql.exception.SqlBuilderException;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 表达式选择项实现
 *
 * @author zeng
 * @version 0.0.4
 */
public class ExpressionSelectItem implements SelectItem {

  /** 字段别名分割正则模式 */
  private static final Pattern FIELD_ALIAS_SPLITTING_PATTERN =
      Pattern.compile("( +as +(?=\\w+$))|( +(?=\\w+$))", Pattern.CASE_INSENSITIVE);

  /** 字段表达式 */
  private final String expression;

  /** 字段名别 */
  private final String fieldAlias;

  public String getExpression() {
    return expression;
  }

  public String getFieldAlias() {
    return fieldAlias;
  }

  /**
   * 构建查询字段值
   *
   * @param expression 查询字段表达式
   */
  public ExpressionSelectItem(String expression) {
    if (StrUtils.isBlank(expression)) {
      throw new SqlBuilderException("选择字段表达式不能为空");
    }

    String[] tokens = FIELD_ALIAS_SPLITTING_PATTERN.split(expression);
    if (tokens.length == 0) {
      throw new SqlBuilderException("选择字段表达式不能为空");
    }

    if (tokens.length == 1) {
      this.expression = tokens[0];
      this.fieldAlias = null;
      return;
    }

    this.expression = tokens[0];
    this.fieldAlias = tokens[1];
  }

  /**
   * 构建SQL片段
   *
   * @return SQL片段
   */
  @Override
  public String buildSqlFragment() {
    StringBuilder stringBuilder = new StringBuilder();
    if (StrUtils.isBlank(this.fieldAlias)) {
      stringBuilder.append(NamingCase.toUnderlineCase(this.expression));
    } else {
      stringBuilder
          .append(NamingCase.toUnderlineCase(this.expression))
          .append(SqlPool.AS_KEYWORD_WITH_SPACE)
          .append(NamingCase.toUnderlineCase(this.fieldAlias));
    }

    return stringBuilder.toString();
  }
}
