package org.elsfs.tool.sql.singular.facade;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 实体关系描述
 *
 * @author zeng
 * @since 0.0.4
 */
public class EntityRelationDescriptor {

  /** 实体类 */
  private final Class<?> entityClass;

  /** 外键字段名称 */
  private final String foreignKeyField;

  /** 连接字段名称 */
  private final String joinKeyField;

  /** 关联选择字段 */
  private final List<String> selectionFields;

  public Class<?> getEntityClass() {
    return entityClass;
  }

  public String getForeignKeyField() {
    return foreignKeyField;
  }

  public String getJoinKeyField() {
    return joinKeyField;
  }

  public List<String> getSelectionFields() {
    return selectionFields;
  }

  /**
   * 构造函数
   *
   * @param entityClass 实体类
   * @param foreignKeyField 外键字段
   * @param joinKeyField 连接字段
   * @param selectionFields 选择字段
   */
  public EntityRelationDescriptor(
      Class<?> entityClass,
      String foreignKeyField,
      String joinKeyField,
      List<String> selectionFields) {
    this.entityClass = entityClass;
    this.foreignKeyField = foreignKeyField;
    this.joinKeyField = joinKeyField;
    this.selectionFields = selectionFields;
  }

  /**
   * 创建实体关联描述对象
   *
   * @param entityClass 实体类
   * @param foreignKeyField 外键字段
   * @param selectionFields 选择字段
   * @return 实体关系描述对象
   */
  public static EntityRelationDescriptor create(
      Class<?> entityClass, String foreignKeyField, String... selectionFields) {
    return new EntityRelationDescriptor(
        entityClass, foreignKeyField, null, Arrays.asList(selectionFields));
  }

  /**
   * 创建实体关联描述对象
   *
   * @param entityClass 实体类
   * @param foreignKeyField 外键字段
   * @param selectionFields 选择字段
   * @return 实体关系描述对象
   */
  public static EntityRelationDescriptor create(
      Class<?> entityClass, Enum<?> foreignKeyField, String... selectionFields) {
    return new EntityRelationDescriptor(
        entityClass, foreignKeyField.name(), null, Arrays.asList(selectionFields));
  }

  /**
   * 创建一个实体关联描述符构建器
   *
   * @param entityClass 实体类
   * @param foreignKeyField 主表中外键字段名称
   * @return 构建器
   */
  public static EntityRelationDescriptorBuilder builder(
      Class<?> entityClass, String foreignKeyField) {
    return new EntityRelationDescriptorBuilder(entityClass, foreignKeyField);
  }

  /**
   * 创建一个实体关联描述符构建器
   *
   * @param entityClass 实体类
   * @param foreignKeyField 主表中外键字段名称枚举
   * @return 构建器
   */
  public static EntityRelationDescriptorBuilder builder(
      Class<?> entityClass, Enum<?> foreignKeyField) {
    return new EntityRelationDescriptorBuilder(entityClass, foreignKeyField.name());
  }

  /** 实体关联描述符构建器 */
  public static class EntityRelationDescriptorBuilder {
    /** 关联实体类 */
    private final Class<?> entityClass;

    /** 外键字段名称 */
    private final String foreignKeyField;

    /** 关联选择字段 */
    private final List<String> selectionFields = new ArrayList<>();

    public EntityRelationDescriptorBuilder(Class<?> entityClass, String foreignKeyField) {
      this.entityClass = entityClass;
      this.foreignKeyField = foreignKeyField;
    }

    /** 连接字段名称 */
    private String joinKeyField;

    /**
     * 设置连接字段
     *
     * @param field 字段名称
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder joinKeyField(String field) {
      this.joinKeyField = field;
      return this;
    }

    /**
     * 设置连接字段
     *
     * @param field 字段名称枚举
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder joinKeyField(Enum<?> field) {
      this.joinKeyField = field.name();
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder select(String field) {
      this.selectionFields.add(field);
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称枚举
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder select(Enum<?> field) {
      this.selectionFields.add(field.name());
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称
     * @param alias 字段别名
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder selectAs(String field, String alias) {
      this.selectionFields.add(field + SqlPool.AS_KEYWORD_WITH_SPACE + alias);
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称枚举
     * @param alias 字段别名枚举
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder selectAs(Enum<?> field, Enum<?> alias) {
      this.selectionFields.add(field.name() + SqlPool.AS_KEYWORD_WITH_SPACE + alias.name());
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称枚举
     * @param alias 字段别名
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder selectAs(Enum<?> field, String alias) {
      this.selectionFields.add(field.name() + SqlPool.AS_KEYWORD_WITH_SPACE + alias);
      return this;
    }

    /**
     * 选择字段
     *
     * @param field 字段名称
     * @param alias 字段别名枚举
     * @return 构建器
     */
    public EntityRelationDescriptorBuilder selectAs(String field, Enum<?> alias) {
      this.selectionFields.add(field + SqlPool.AS_KEYWORD_WITH_SPACE + alias.name());
      return this;
    }

    /**
     * 构建实体关联描述符
     *
     * @return 实体关联描述符
     */
    public EntityRelationDescriptor build() {
      return new EntityRelationDescriptor(
          this.entityClass, this.foreignKeyField, this.joinKeyField, this.selectionFields);
    }
  }
}
