package org.elsfs.tool.sql.abs;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Consumer;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.common.FieldEnumResolverSupport;
import org.elsfs.tool.sql.common.HierarchyTableAliasManager;
import org.elsfs.tool.sql.common.SimpleTableAliasManager;
import org.elsfs.tool.sql.condition.AndConditionItem;
import org.elsfs.tool.sql.condition.BetweenBeginValueConditionItem;
import org.elsfs.tool.sql.condition.BetweenEndValueConditionItem;
import org.elsfs.tool.sql.condition.BetweenFieldConditionItem;
import org.elsfs.tool.sql.condition.BetweenValueConditionItem;
import org.elsfs.tool.sql.condition.CompareOperator;
import org.elsfs.tool.sql.condition.CompositeConditionItem;
import org.elsfs.tool.sql.condition.ConditionItem;
import org.elsfs.tool.sql.condition.ExpressionConditionItem;
import org.elsfs.tool.sql.condition.FieldConditionItem;
import org.elsfs.tool.sql.condition.FieldSubQueryConditionItem;
import org.elsfs.tool.sql.condition.FieldValueConditionItem;
import org.elsfs.tool.sql.condition.InConditionItem;
import org.elsfs.tool.sql.condition.InSubQueryConditionItem;
import org.elsfs.tool.sql.condition.IsNotNullConditionItem;
import org.elsfs.tool.sql.condition.IsNotNullSubQueryConditionItem;
import org.elsfs.tool.sql.condition.IsNullConditionItem;
import org.elsfs.tool.sql.condition.IsNullSubQueryConditionItem;
import org.elsfs.tool.sql.condition.LikeConditionItem;
import org.elsfs.tool.sql.condition.LikeType;
import org.elsfs.tool.sql.condition.NotBetweenBeginValueConditionItem;
import org.elsfs.tool.sql.condition.NotBetweenEndValueConditionItem;
import org.elsfs.tool.sql.condition.NotBetweenFieldConditionItem;
import org.elsfs.tool.sql.condition.NotBetweenValueConditionItem;
import org.elsfs.tool.sql.condition.NotInConditionItem;
import org.elsfs.tool.sql.condition.NotInSubQueryConditionItem;
import org.elsfs.tool.sql.condition.NotLikeConditionItem;
import org.elsfs.tool.sql.condition.OrConditionItem;
import org.elsfs.tool.sql.interfaces.Condition;
import org.elsfs.tool.sql.interfaces.SqlParameterManager;
import org.elsfs.tool.sql.interfaces.SqlParameterManagerAware;
import org.elsfs.tool.sql.interfaces.TableAliasManager;
import org.elsfs.tool.sql.interfaces.TableAliasManagerAware;
import org.elsfs.tool.sql.utils.CastUtils;

/**
 * 抽象条件实现
 *
 * @author zeng
 * @version 0.0.4
 */
public abstract class AbstractCondition<C extends Condition<C>> extends CompositeSqlFragment
    implements Condition<C>, SqlParameterManagerAware, TableAliasManagerAware {

  /** 子类实现 */
  protected final C childThis = CastUtils.cast(this);

  @Override
  public C eq(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(leftField, rightField, CompareOperator.EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C eq(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField, standardSelectSql.buildSqlFragment(), CompareOperator.EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C eqValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(), leftField, rightValue, CompareOperator.EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C ne(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(leftField, rightField, CompareOperator.NOT_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 不相等条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C ne(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField, standardSelectSql.buildSqlFragment(), CompareOperator.NOT_EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C neValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(),
                  leftField,
                  rightValue,
                  CompareOperator.NOT_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 小于条件
   *
   * @param leftField 左边字段名称
   * @param rightField 右边字段名称
   * @return 具体实现
   */
  @Override
  public C lt(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(leftField, rightField, CompareOperator.LESS_THAN)));
    }
    return this.childThis;
  }

  /**
   * 小于条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C lt(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField, standardSelectSql.buildSqlFragment(), CompareOperator.LESS_THAN)));
    }
    return this.childThis;
  }

  /**
   * 解析字段枚举名称
   *
   * @param field 字段枚举
   * @return 字段名称
   */
  public String resolveFieldName(Enum<?> field) {
    return FieldEnumResolverSupport.resolveFieldName(this.getTableAliasManager(), field);
  }

  /**
   * 小于条件
   *
   * @param leftField 左边字段名称
   * @param rightValue 右边值
   * @return 具体实现
   */
  @Override
  public C ltValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(),
                  leftField,
                  rightValue,
                  CompareOperator.LESS_THAN)));
    }
    return this.childThis;
  }

  /**
   * 小于等于条件
   *
   * @param leftField 左边字段名称
   * @param rightField 右边字段名称
   * @return 具体实现
   */
  @Override
  public C le(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(leftField, rightField, CompareOperator.LESS_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 小于等于条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C le(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField,
                  standardSelectSql.buildSqlFragment(),
                  CompareOperator.LESS_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C leValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(),
                  leftField,
                  rightValue,
                  CompareOperator.LESS_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  @Override
  public C gt(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(leftField, rightField, CompareOperator.GREATER_THAN)));
    }
    return this.childThis;
  }

  @Override
  public C gt(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField, standardSelectSql.buildSqlFragment(), CompareOperator.GREATER_THAN)));
    }
    return this.childThis;
  }

  /**
   * 大于条件
   *
   * @param leftField 左边字段名称
   * @param rightValue 右边值
   * @return 具体实现
   */
  @Override
  public C gtValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(),
                  leftField,
                  rightValue,
                  CompareOperator.GREATER_THAN)));
    }
    return this.childThis;
  }

  @Override
  public C ge(boolean condition, String leftField, String rightField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldConditionItem(
                  leftField, rightField, CompareOperator.GREATER_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 大于等于条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C ge(boolean condition, String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    if (condition) {
      StandardSelectSql standardSelectSql =
          new StandardSelectSql(
              this.getSqlParameterManager(),
              new HierarchyTableAliasManager(
                  new SimpleTableAliasManager(), this.getTableAliasManager()));
      sqlBuilderConsumer.accept(standardSelectSql);
      this.addConditionItem(
          new AndConditionItem(
              new FieldSubQueryConditionItem(
                  leftField,
                  standardSelectSql.buildSqlFragment(),
                  CompareOperator.GREATER_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 大于等于条件
   *
   * @param leftField 左边字段名称
   * @param rightValue 右边值
   * @return 具体实现
   */
  @Override
  public C geValue(boolean condition, String leftField, Object rightValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new FieldValueConditionItem(
                  this.getSqlParameterManager(),
                  leftField,
                  rightValue,
                  CompareOperator.GREATER_THAN_OR_EQUAL)));
    }
    return this.childThis;
  }

  /**
   * 区间条件
   *
   * @param leftField 左边字段名称
   * @param beginField 区间开始字段名称
   * @param endField 区间结束字段名称
   * @return 具体实现
   */
  @Override
  public C between(boolean condition, String leftField, String beginField, String endField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(new BetweenFieldConditionItem(leftField, beginField, endField)));
    }
    return this.childThis;
  }

  /**
   * 区间条件
   *
   * @param leftField 左边字段名称
   * @param beginValue 区间开始值
   * @param endField 区间结束字段名称
   * @return 具体实现
   */
  @Override
  public C betweenBeginValue(
      boolean condition, String leftField, Object beginValue, String endField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new BetweenBeginValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginValue, endField)));
    }
    return this.childThis;
  }

  /**
   * 区间条件
   *
   * @param leftField 左边字段名称
   * @param beginField 区间开始字段名称
   * @param endValue 区间结束值
   * @return 具体实现
   */
  @Override
  public C betweenEndValue(
      boolean condition, String leftField, String beginField, Object endValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new BetweenEndValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginField, endValue)));
    }
    return this.childThis;
  }

  /**
   * 区间条件
   *
   * @param leftField 左边字段名称
   * @param beginValue 区间开始值
   * @param endValue 区间结束值
   * @return 具体实现
   */
  @Override
  public C betweenValue(boolean condition, String leftField, Object beginValue, Object endValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new BetweenValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginValue, endValue)));
    }
    return this.childThis;
  }

  /**
   * 非区间条件
   *
   * @param leftField 左边字段名称
   * @param beginField 区间开始字段名称
   * @param endField 区间结束字段名称
   * @return 具体实现
   */
  @Override
  public C notBetween(boolean condition, String leftField, String beginField, String endField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(new NotBetweenFieldConditionItem(leftField, beginField, endField)));
    }
    return this.childThis;
  }

  /**
   * 非区间条件
   *
   * @param leftField 左边字段名称
   * @param beginValue 区间开始值
   * @param endField 区间结束字段名称
   * @return 具体实现
   */
  @Override
  public C notBetweenBeginValue(
      boolean condition, String leftField, Object beginValue, String endField) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new NotBetweenBeginValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginValue, endField)));
    }
    return this.childThis;
  }

  /**
   * 非区间条件
   *
   * @param leftField 左边字段名称
   * @param beginField 区间开始字段名称
   * @param endValue 区间结束值
   * @return 具体实现
   */
  @Override
  public C notBetweenEndValue(
      boolean condition, String leftField, String beginField, Object endValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new NotBetweenEndValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginField, endValue)));
    }
    return this.childThis;
  }

  /**
   * 非区间条件
   *
   * @param leftField 左边字段名称
   * @param beginValue 区间开始值
   * @param endValue 区间结束值
   * @return 具体实现
   */
  @Override
  public C notBetweenValue(
      boolean condition, String leftField, Object beginValue, Object endValue) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new NotBetweenValueConditionItem(
                  this.getSqlParameterManager(), leftField, beginValue, endValue)));
    }
    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  @Override
  public C in(boolean condition, String leftField, String... rightFields) {
    if (condition) {
      this.addConditionItem(
          new AndConditionItem(
              new InConditionItem(
                  this.getSqlParameterManager(), leftField, Arrays.asList(rightFields), null)));
    }

    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  @Override
  public C in(String leftField, Collection<String> rightFields) {
    this.addConditionItem(
        new AndConditionItem(
            new InConditionItem(this.getSqlParameterManager(), leftField, rightFields, null)));

    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C in(String leftField, Collection<String> rightFields, Collection<?> rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new InConditionItem(
                this.getSqlParameterManager(), leftField, rightFields, rightValues)));

    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C in(String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(
            this.getSqlParameterManager(),
            new HierarchyTableAliasManager(
                new SimpleTableAliasManager(), this.getTableAliasManager()));
    sqlBuilderConsumer.accept(standardSelectSql);
    this.addConditionItem(
        new AndConditionItem(
            new InSubQueryConditionItem(leftField, standardSelectSql.buildSqlFragment())));

    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C inValue(String leftField, Object... rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new InConditionItem(
                this.getSqlParameterManager(), leftField, null, Arrays.asList(rightValues))));

    return this.childThis;
  }

  /**
   * 范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C inValue(String leftField, Collection<?> rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new InConditionItem(this.getSqlParameterManager(), leftField, null, rightValues)));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  @Override
  public C notIn(String leftField, String... rightFields) {
    this.addConditionItem(
        new AndConditionItem(
            new NotInConditionItem(
                this.getSqlParameterManager(), leftField, Arrays.asList(rightFields), null)));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @return 具体实现
   */
  @Override
  public C notIn(String leftField, Collection<String> rightFields) {
    this.addConditionItem(
        new AndConditionItem(
            new NotInConditionItem(this.getSqlParameterManager(), leftField, rightFields, null)));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param rightFields 范围字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C notIn(String leftField, Collection<String> rightFields, Collection<?> rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new NotInConditionItem(
                this.getSqlParameterManager(), leftField, rightFields, rightValues)));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C notIn(String leftField, Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(
            this.getSqlParameterManager(),
            new HierarchyTableAliasManager(
                new SimpleTableAliasManager(), this.getTableAliasManager()));
    sqlBuilderConsumer.accept(standardSelectSql);
    this.addConditionItem(
        new AndConditionItem(
            new NotInSubQueryConditionItem(leftField, standardSelectSql.buildSqlFragment())));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C notInValue(String leftField, Object... rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new NotInConditionItem(
                this.getSqlParameterManager(), leftField, null, Arrays.asList(rightValues))));

    return this.childThis;
  }

  /**
   * 非范围条件
   *
   * @param leftField 左边字段名称
   * @param rightValues 范围值
   * @return 具体实现
   */
  @Override
  public C notInValue(String leftField, Collection<?> rightValues) {
    this.addConditionItem(
        new AndConditionItem(
            new NotInConditionItem(this.getSqlParameterManager(), leftField, null, rightValues)));

    return this.childThis;
  }

  /**
   * 为空条件
   *
   * @param field 字段名称
   * @return 具体实现
   */
  @Override
  public C isNull(String field) {
    this.addConditionItem(new AndConditionItem(new IsNullConditionItem(field)));

    return this.childThis;
  }

  /**
   * 为空条件
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C isNull(Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(
            this.getSqlParameterManager(),
            new HierarchyTableAliasManager(
                new SimpleTableAliasManager(), this.getTableAliasManager()));
    sqlBuilderConsumer.accept(standardSelectSql);
    this.addConditionItem(
        new AndConditionItem(
            new IsNullSubQueryConditionItem(standardSelectSql.buildSqlFragment())));

    return this.childThis;
  }

  /**
   * 非空条件
   *
   * @param field 字段名称
   * @return 具体实现
   */
  @Override
  public C isNotNull(String field) {
    this.addConditionItem(new AndConditionItem(new IsNotNullConditionItem(field)));

    return this.childThis;
  }

  /**
   * 非空条件
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C isNotNull(Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(
            this.getSqlParameterManager(),
            new HierarchyTableAliasManager(
                new SimpleTableAliasManager(), this.getTableAliasManager()));
    sqlBuilderConsumer.accept(standardSelectSql);
    this.addConditionItem(
        new AndConditionItem(
            new IsNotNullSubQueryConditionItem(standardSelectSql.buildSqlFragment())));

    return this.childThis;
  }

  /**
   * 模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C like(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new LikeConditionItem(this.getSqlParameterManager(), LikeType.LIKE, field, value)));

    return this.childThis;
  }

  /**
   * 模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C like(Enum<?> field, String value) {
    return this.like(this.resolveFieldName(field), value);
  }

  /**
   * 模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C like(String tableAlias, Enum<?> field, String value) {
    return this.like(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 左模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeLeft(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new LikeConditionItem(
                this.getSqlParameterManager(), LikeType.LIKE_LEFT, field, value)));

    return this.childThis;
  }

  /**
   * 左模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeLeft(Enum<?> field, String value) {
    return this.likeLeft(this.resolveFieldName(field), value);
  }

  /**
   * 左模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeLeft(String tableAlias, Enum<?> field, String value) {
    return this.likeLeft(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 右模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeRight(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new LikeConditionItem(
                this.getSqlParameterManager(), LikeType.LIKE_RIGHT, field, value)));

    return this.childThis;
  }

  /**
   * 右模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeRight(Enum<?> field, String value) {
    return this.likeRight(this.resolveFieldName(field), value);
  }

  /**
   * 右模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C likeRight(String tableAlias, Enum<?> field, String value) {
    return this.likeRight(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 非模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLike(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new NotLikeConditionItem(this.getSqlParameterManager(), LikeType.LIKE, field, value)));

    return this.childThis;
  }

  /**
   * 非模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLike(Enum<?> field, String value) {
    return this.notLike(this.resolveFieldName(field), value);
  }

  /**
   * 非模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLike(String tableAlias, Enum<?> field, String value) {
    return this.notLike(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 非左模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeLeft(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new NotLikeConditionItem(
                this.getSqlParameterManager(), LikeType.LIKE_LEFT, field, value)));

    return this.childThis;
  }

  /**
   * 非左模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeLeft(Enum<?> field, String value) {
    return this.notLikeLeft(this.resolveFieldName(field), value);
  }

  /**
   * 非左模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeLeft(String tableAlias, Enum<?> field, String value) {
    return this.notLikeLeft(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 非右模糊匹配条件
   *
   * @param field 字段名称
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeRight(String field, String value) {
    this.addConditionItem(
        new AndConditionItem(
            new NotLikeConditionItem(
                this.getSqlParameterManager(), LikeType.LIKE_RIGHT, field, value)));

    return this.childThis;
  }

  /**
   * 非右模糊匹配条件
   *
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeRight(Enum<?> field, String value) {
    return this.notLikeRight(this.resolveFieldName(field), value);
  }

  /**
   * 非右模糊匹配条件
   *
   * @param tableAlias 字段表别名
   * @param field 字段枚举
   * @param value 匹配值
   * @return 具体实现
   */
  @Override
  public C notLikeRight(String tableAlias, Enum<?> field, String value) {
    return this.notLikeRight(this.resolveFieldName(tableAlias, field), value);
  }

  /**
   * 表达式匹配条件
   *
   * @param expression 表达式
   * @param args 表达式参数
   * @return 具体实现
   */
  @Override
  public C expression(String expression, Object... args) {
    this.addConditionItem(new AndConditionItem(new ExpressionConditionItem(expression, args)));

    return this.childThis;
  }

  /**
   * 嵌套条件
   *
   * @return 条件嵌套器
   */
  @Override
  public AbstractNestCondition<C> nest() {
    NestConditionImpl nestCondition = new NestConditionImpl();
    this.addConditionItem(
        new AndConditionItem(new CompositeConditionItem(nestCondition.getConditionItems())));

    return nestCondition;
  }

  /**
   * 开始或条件
   *
   * @return 或条件构建器
   */
  @Override
  public AbstractOrCondition<C> or() {
    OrConditionImpl orCondition = new OrConditionImpl();
    this.addConditionItem(
        new OrConditionItem(new CompositeConditionItem(orCondition.getConditionItems())));

    return orCondition;
  }

  /**
   * 添加一个条件
   *
   * @param conditionItem 条件
   */
  protected abstract void addConditionItem(ConditionItem conditionItem);

  /**
   * 获取SQL参数管理器
   *
   * @return SQL参数管理器
   */
  @Override
  public SqlParameterManager getSqlParameterManager() {
    return null;
  }

  /**
   * 获取表别名管理器
   *
   * @return 表名别管理器
   */
  @Override
  public TableAliasManager getTableAliasManager() {
    return null;
  }

  /** 嵌套条件构建实现 */
  public class NestConditionImpl extends AbstractNestCondition<C> {

    /**
     * 结束嵌套条件
     *
     * @return 上级对象
     */
    @Override
    public C endNest() {
      return AbstractCondition.this.childThis;
    }

    /**
     * 获取SQL参数管理器
     *
     * @return SQL参数管理器
     */
    @Override
    public SqlParameterManager getSqlParameterManager() {
      return AbstractCondition.this.getSqlParameterManager();
    }

    /**
     * 获取表别名管理器
     *
     * @return 表名别管理器
     */
    @Override
    public TableAliasManager getTableAliasManager() {
      return AbstractCondition.this.getTableAliasManager();
    }
  }

  /** Or条件构建实现 */
  public class OrConditionImpl extends AbstractOrCondition<C> {

    /**
     * 结束或条件
     *
     * @return 上级对象
     */
    @Override
    public C endOr() {
      return AbstractCondition.this.childThis;
    }

    /**
     * 获取SQL参数管理器
     *
     * @return SQL参数管理器
     */
    @Override
    public SqlParameterManager getSqlParameterManager() {
      return AbstractCondition.this.getSqlParameterManager();
    }

    /**
     * 获取表别名管理器
     *
     * @return 表名别管理器
     */
    @Override
    public TableAliasManager getTableAliasManager() {
      return AbstractCondition.this.getTableAliasManager();
    }
  }
}
