package org.elsfs.tool.sql.abs;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.elsfs.tool.core.text.StrFormatter;
import org.elsfs.tool.sql.condition.ConditionItem;
import org.elsfs.tool.sql.interfaces.SqlFragment;
import org.elsfs.tool.sql.interfaces.TokenSqlFragment;
import org.elsfs.tool.sql.limit.LimitItem;
import org.elsfs.tool.sql.order.CompositeOrderItem;
import org.elsfs.tool.sql.order.OrderItem;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 可排序条件构建器抽象实现
 *
 * @param <C> 子类具体类型
 * @author zeng
 * @since 0.0.4
 */
public abstract class AbstractSortableConditionBuilder<
        C extends AbstractSortableConditionBuilder<C>>
    extends AbstractSortableCondition<C> {

  /** 条件项列表 */
  private final List<ConditionItem> conditionItems = new ArrayList<>();

  /** 聚合条件项 */
  private final List<OrderItem> orderItems = new ArrayList<>();

  /** 限制数量项 */
  private LimitItem limitItem;

  /**
   * 添加一个条件
   *
   * @param conditionItem 条件
   */
  @Override
  protected void addConditionItem(ConditionItem conditionItem) {
    this.conditionItems.add(conditionItem);
  }

  /**
   * 添加排序项
   *
   * @param orderItem 排序项
   */
  @Override
  protected void addOrderItem(OrderItem orderItem) {
    this.orderItems.add(orderItem);
  }

  /**
   * 设置限制数量项
   *
   * @param limitItem 限制数量项
   */
  @Override
  protected void setLimitItem(LimitItem limitItem) {
    this.limitItem = limitItem;
  }

  /** 构建SQL之前处理 */
  @Override
  protected void beforeBuild() {
    if (!this.conditionItems.isEmpty()) {
      this.addSqlFragment(
          new TokenSqlFragment(
              StrFormatter.format(
                  "WHERE {}",
                  ConditionItem.stripHeaderKeywordStatic(
                      this.conditionItems.stream()
                          .map(SqlFragment::buildSqlFragment)
                          .collect(Collectors.joining(SqlPool.SPACE))))));
    }
    if (!this.orderItems.isEmpty()) {
      this.addSqlFragment(new CompositeOrderItem(this.orderItems));
    }
    if (this.limitItem != null) {
      this.addSqlFragment(this.limitItem);
    }
  }
}
