package org.elsfs.tool.sql.abs;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.common.SimpleTableAliasManager;
import org.elsfs.tool.sql.interfaces.SqlFragment;
import org.elsfs.tool.sql.interfaces.SqlParameterManagerAware;
import org.elsfs.tool.sql.interfaces.UnionSql;
import org.elsfs.tool.sql.interfaces.UnionType;
import org.elsfs.tool.sql.union.StandardUnionItem;
import org.elsfs.tool.sql.union.UnionItem;
import org.elsfs.tool.sql.utils.CastUtils;
import org.elsfs.tool.sql.utils.SqlPool;

/**
 * 抽象的联合SQL实现
 *
 * @param <C> 子类具体实现
 * @author zeng
 * @since 0.0.4
 */
public abstract class AbstractUnionSql<C extends AbstractUnionSql<C>>
    implements UnionSql<C>, SqlParameterManagerAware {

  /** 具体子类实现引用 */
  protected final C childThis = CastUtils.cast(this);

  /** 联合项列表 */
  private final List<UnionItem> unionItems = new ArrayList<>();

  /**
   * 联合（去重和排序）
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C union(Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(this.getSqlParameterManager(), new SimpleTableAliasManager());
    sqlBuilderConsumer.accept(standardSelectSql);
    this.unionItems.add(new StandardUnionItem(standardSelectSql, UnionType.UNION));

    return this.childThis;
  }

  /**
   * 粗暴联合
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @return 具体实现
   */
  @Override
  public C unionAll(Consumer<StandardSelectSql> sqlBuilderConsumer) {
    StandardSelectSql standardSelectSql =
        new StandardSelectSql(this.getSqlParameterManager(), new SimpleTableAliasManager());
    sqlBuilderConsumer.accept(standardSelectSql);
    this.unionItems.add(new StandardUnionItem(standardSelectSql, UnionType.UNION_ALL));

    return this.childThis;
  }

  /**
   * 构建SQL片段
   *
   * @return SQL片段
   */
  @Override
  public String buildSqlFragment() {
    if (this.unionItems.isEmpty()) {
      return SqlPool.EMPTY;
    }

    return UnionItem.stripHeaderKeyword(
        this.unionItems.stream()
            .map(SqlFragment::buildSqlFragment)
            .collect(Collectors.joining(SqlPool.SPACE)));
  }
}
