package org.elsfs.tool.sql.interfaces.select;

import java.util.Collection;
import java.util.function.Consumer;
import org.elsfs.tool.sql.builder.StandardSelectSql;
import org.elsfs.tool.sql.utils.CastUtils;

/**
 * 可选择接口
 *
 * @param <C> 子类实现
 * @author zeng
 * @since 0.0.4
 */
public interface Selectable<C extends Selectable<C>> {

  /**
   * 选择SQL字段
   *
   * @param fields SQL字段名称
   * @return 具体实现
   */
  C select(String... fields);

  /**
   * 选择SQL字段
   *
   * @param fields SQL字段枚举
   * @return 具体实现
   */
  C select(Enum<?>... fields);

  /**
   * 选择SQL字段
   *
   * @param fields SQL字段
   * @return 具体实现
   */
  C select(Collection<String> fields);

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param fields SQL字段
   * @return 具体实现
   */
  default C select(boolean condition, String... fields) {
    if (condition) {
      return this.select(fields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param fields SQL字段枚举
   * @return 具体实现
   */
  default C select(boolean condition, Enum<?>... fields) {
    if (condition) {
      return this.select(fields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param fields SQL字段
   * @return 具体实现
   */
  default C select(boolean condition, Collection<String> fields) {
    if (condition) {
      return this.select(fields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段表达式
   *
   * @param expressions SQL字段表达式
   * @return 具体实现
   */
  C selectExpression(String... expressions);

  /**
   * 选择SQL字段表达式
   *
   * @param expressions SQL字段表达式
   * @return 具体实现
   */
  C selectExpression(Collection<String> expressions);

  /**
   * 选择SQL字段表达式
   *
   * @param condition 条件
   * @param expressions SQL字段表达式
   * @return 具体实现
   */
  default C selectExpression(boolean condition, String... expressions) {
    if (condition) {
      return this.selectExpression(expressions);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段表达式
   *
   * @param condition 条件
   * @param expressions SQL字段表达式
   * @return 具体实现
   */
  default C selectExpression(boolean condition, Collection<String> expressions) {
    if (condition) {
      return this.selectExpression(expressions);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param field SQL字段
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  C selectAs(String field, String fieldAlias);

  /**
   * 选择SQL字段
   *
   * @param field SQL字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  C selectAs(Enum<?> field, String fieldAlias);

  /**
   * 选择SQL字段
   *
   * @param field SQL字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  C selectAs(Enum<?> field, Enum<?> fieldAlias);

  /**
   * 选择子查询结果
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  C selectAs(Consumer<StandardSelectSql> sqlBuilderConsumer, String fieldAlias);

  /**
   * 选择子查询结果
   *
   * @param sqlBuilderConsumer SQL构建器消费器
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  C selectAs(Consumer<StandardSelectSql> sqlBuilderConsumer, Enum<?> fieldAlias);

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param field SQL字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  default C selectAs(boolean condition, Enum<?> field, String fieldAlias) {
    if (condition) {
      return this.selectAs(field, fieldAlias);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param field SQL字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  default C selectAs(boolean condition, Enum<?> field, Enum<?> fieldAlias) {
    if (condition) {
      return this.selectAs(field, fieldAlias);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择子查询结果
   *
   * @param condition 条件
   * @param sqlBuilderConsumer SQL构建器消费器
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  default C selectAs(
      boolean condition, Consumer<StandardSelectSql> sqlBuilderConsumer, String fieldAlias) {
    if (condition) {
      return this.selectAs(sqlBuilderConsumer, fieldAlias);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择子查询结果
   *
   * @param condition 条件
   * @param sqlBuilderConsumer SQL构建器消费器
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  default C selectAs(
      boolean condition, Consumer<StandardSelectSql> sqlBuilderConsumer, Enum<?> fieldAlias) {
    if (condition) {
      return this.selectAs(sqlBuilderConsumer, fieldAlias);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param condition 条件
   * @param fields SQL字段枚举
   * @return 具体实现
   */
  default C selectEnum(boolean condition, Collection<Enum<?>> fields) {
    if (condition) {
      return this.selectEnum(fields);
    }

    return CastUtils.cast(this);
  }

  /**
   * 选择SQL字段
   *
   * @param fields SQL字段枚举
   * @return 具体实现
   */
  C selectEnum(Collection<Enum<?>> fields);
}
