package org.elsfs.tool.sql.service;

import java.util.Collection;
import java.util.List;
import org.elsfs.tool.sql.singular.facade.DbActionFactory;

/**
 * 增删改查Service
 *
 * @author zeng
 * @since
 */
public interface CrudService<E> extends DbActionFactory<E> {

  /**
   * 创建一个实体对象实例
   *
   * @return 实体对象
   */
  E newEntity();

  /**
   * 获取实体类
   *
   * @return 实体类
   */
  Class<E> getEntityClass();

  /**
   * 判断指定ID的数据是否存在
   *
   * @param id ID
   * @return 结果bool
   */
  boolean existsById(String id);

  /**
   * 根据ID获取一条数据
   *
   * @param id ID
   * @return 实体对象
   */
  E fetchOne(String id);

  /**
   * 根据多个ID获取数据
   *
   * @param ids ID集合
   * @return 实体对象列表
   */
  List<E> fetchList(Collection<String> ids);

  /**
   * 获取所有数据
   *
   * @return 实体对象列表
   */
  List<E> fetchList();

  /**
   * 创建一个实体
   *
   * @param entity 实体对象
   */
  void create(E entity);

  /**
   * 修改一个实体
   *
   * @param entity 实体对象
   */
  void modify(E entity);

  /**
   * 创建或修改一个实体
   *
   * @param entity 实体对象
   */
  void createOrModify(E entity);

  /**
   * 批量插入创建数据
   *
   * @param entities 实体对象集合
   */
  void batchCreate(List<E> entities);

  /**
   * 批量修改数据
   *
   * @param entities 实体对象集合
   */
  void batchModify(List<E> entities);

  /**
   * 删除一个实体（记录）
   *
   * @param entity 实体对象
   */
  void delete(E entity);

  /**
   * 根据ID删除一条记录
   *
   * @param id ID
   */
  void deleteById(String id);

  /**
   * 根据ID集合批量删除
   *
   * @param ids ID集合
   */
  void batchDeleteByIds(Collection<String> ids);

  /**
   * 批量删除
   *
   * @param entities 实体对象集合
   */
  void batchDelete(List<E> entities);
}
