package org.elsfs.tool.sql.singular.facade;

import java.util.List;
import org.elsfs.tool.core.page.IPage;
import org.elsfs.tool.sql.resolve.ResolveFieldName;

/**
 * 查询操作
 *
 * @param <C> 子类类型
 * @author zeng
 * @since 0.0.4
 */
public interface QueryOperations<C extends QueryOperations<C>>
    extends ConditionOperations<C>, LimitOperations<C>, ResolveFieldName {

  /**
   * 选择字段
   *
   * @param fields 字段数组
   * @return 具体实现
   */
  C select(String... fields);

  /**
   * 选择字段
   *
   * @param fields 字段枚举数组
   * @return 具体实现
   */
  C select(Enum<?>... fields);

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return 具体实现
   */
  C select(boolean condition, String... fields);

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return 具体实现
   */
  C select(boolean condition, Enum<?>... fields);

  /**
   * 选择字段
   *
   * @param field 字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  C selectAs(Enum<?> field, String fieldAlias);

  /**
   * 选择字段
   *
   * @param field 字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  C selectAs(Enum<?> field, Enum<?> fieldAlias);

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @param fieldAlias 字段别名
   * @return 具体实现
   */
  C selectAs(boolean condition, Enum<?> field, String fieldAlias);

  /**
   * 选择字段
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @param fieldAlias 字段别名枚举
   * @return 具体实现
   */
  C selectAs(boolean condition, Enum<?> field, Enum<?> fieldAlias);

  /**
   * 设置分页信息
   *
   * @param pageable 可分页对象
   * @return 具体实现
   */
  C pageable(IPage<?> pageable);

  /**
   * 设置分页信息
   *
   * @param condition 执行条件
   * @param page 可分页对象
   * @return 具体实现
   */
  C pageable(boolean condition, IPage<?> page);

  /**
   * 限制查询数据
   *
   * @param condition 执行条件
   * @param offset 开始位置
   * @param num 数量
   * @return 具体实现
   */
  C limit(boolean condition, long offset, long num);

  /**
   * 限制查询数据
   *
   * @param offset 开始位置
   * @param num 数量
   * @return 具体实现
   */
  C limit(long offset, long num);

  /**
   * 偏移数量
   *
   * @param num 数量
   * @return 具体实现
   */
  C offset(long num);

  /**
   * 偏移数量
   *
   * @param condition 执行条件
   * @param num 数量
   * @return 具体实现
   */
  C offset(boolean condition, long num);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param condition 执行条件
   * @param field 单个字段
   * @return children
   */
  C groupBy(boolean condition, String field);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param condition 执行条件
   * @param field 单个字段枚举
   * @return children
   */
  C groupBy(boolean condition, Enum<?> field);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param field 单个字段
   * @return children
   */
  C groupBy(String field);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy("id")
   *
   * @param field 单个字段枚举
   * @return children
   */
  default C groupBy(Enum<?> field) {
    return this.groupBy(this.resolveFieldName(field));
  }

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  C groupBy(List<String> fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  C groupBy(boolean condition, List<String> fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  C groupBy(boolean condition, String... fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  C groupBy(boolean condition, Enum<?>... fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  C groupBy(String... fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  C groupBy(Enum<?>... fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  C groupByEnum(boolean condition, List<Enum<?>> fields);

  /**
   * 分组：GROUP BY 字段, ...
   *
   * <p>例: groupBy(Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  C groupByEnum(List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param condition 执行条件
   * @param field 单个字段
   * @return children
   */
  C orderByAsc(boolean condition, String field);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param condition 执行条件
   * @param field 单个字段枚举
   * @return children
   */
  C orderByAsc(boolean condition, Enum<?> field);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param field 单个字段
   * @return children
   */
  C orderByAsc(String field);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, "id")
   *
   * @param field 单个字段枚举
   * @return children
   */
  default C orderByAsc(Enum<?> field) {
    return this.orderByAsc(this.resolveFieldName(field));
  }

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  C orderByAsc(boolean condition, List<String> fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段数组
   * @return children
   */
  C orderByAsc(boolean condition, String... fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  C orderByAsc(boolean condition, Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  C orderByAsc(String... fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  C orderByAsc(Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段数组
   * @return children
   */
  C orderByAsc(List<String> fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举数组
   * @return children
   */
  C orderByEnumAsc(boolean condition, List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ... ASC
   *
   * <p>例: orderByAsc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举数组
   * @return children
   */
  C orderByEnumAsc(List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段
   * @return children
   */
  C orderBy(boolean isAsc, String field);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段枚举
   * @return children
   */
  default C orderBy(boolean isAsc, Enum<?> field) {
    return this.orderBy(isAsc, this.resolveFieldName(field));
  }

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  C orderBy(boolean isAsc, List<String> fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段
   * @return children
   */
  C orderBy(boolean condition, boolean isAsc, String field);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, "id")
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param field 单个字段枚举
   * @return children
   */
  C orderBy(boolean condition, boolean isAsc, Enum<?> field);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  C orderBy(boolean condition, boolean isAsc, List<String> fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  C orderBy(boolean condition, boolean isAsc, String... fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  C orderBy(boolean condition, boolean isAsc, Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段列表
   * @return children
   */
  C orderBy(boolean isAsc, String... fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  C orderBy(boolean isAsc, Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param condition 执行条件
   * @param field 字段
   * @return children
   */
  C orderByDesc(boolean condition, String field);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param condition 执行条件
   * @param field 字段枚举
   * @return children
   */
  C orderByDesc(boolean condition, Enum<?> field);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param field 字段
   * @return children
   */
  C orderByDesc(String field);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, "id")
   *
   * @param field 字段枚举
   * @return children
   */
  default C orderByDesc(Enum<?> field) {
    return this.orderByDesc(this.resolveFieldName(field));
  }

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段列表
   * @return children
   */
  C orderByDesc(boolean condition, List<String> fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段列表
   * @return children
   */
  C orderByDesc(List<String> fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段列表
   * @return children
   */
  C orderByDesc(boolean condition, String... fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByDesc(boolean condition, Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段列表
   * @return children
   */
  C orderByDesc(String... fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByDesc(Enum<?>... fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByEnum(boolean isAsc, List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ...
   *
   * <p>例: orderBy(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param isAsc 是否是 ASC 排序
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByEnum(boolean condition, boolean isAsc, List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByEnumDesc(List<Enum<?>> fields);

  /**
   * 排序：ORDER BY 字段, ... DESC
   *
   * <p>例: orderByDesc(true, Arrays.asList("id", "name"))
   *
   * @param condition 执行条件
   * @param fields 字段枚举列表
   * @return children
   */
  C orderByEnumDesc(boolean condition, List<Enum<?>> fields);

  /**
   * HAVING ( sql语句 )
   *
   * <p>例1: having("sum(age) &gt; 10")
   *
   * <p>例2: having("sum(age) &gt; {0}", 10)
   *
   * @param sqlValue sql 语句
   * @param params 参数数组
   * @return children
   */
  C having(String sqlValue, Object... params);

  /**
   * HAVING ( sql语句 )
   *
   * <p>例1: having("sum(age) &gt; 10")
   *
   * <p>例2: having("sum(age) &gt; {0}", 10)
   *
   * @param condition 执行条件
   * @param sqlValue sql 语句
   * @param params 参数数组
   * @return children
   */
  C having(boolean condition, String sqlValue, Object... params);
}
