package org.elsfs.tool.core.util;

import org.elsfs.tool.core.text.CharUtil;

/**
 * {@link CharSequence} 相关工具类封装
 *
 * @author zeng
 * @since 0.0.3
 */
public class CharSequenceUtil {

  /** 字符串常量：空字符串 {@code ""} */
  public static final String EMPTY = "";

  /**
   * 字符串是否为空白，空白的定义如下：
   *
   * <ol>
   *   <li>{@code null}
   *   <li>空字符串：{@code ""}
   *   <li>空格、全角空格、制表符、换行符，等不可见字符
   * </ol>
   *
   * <p>例：
   *
   * <ul>
   *   <li>{@code StrUtil.isBlank(null) // true}
   *   <li>{@code StrUtil.isBlank("") // true}
   *   <li>{@code StrUtil.isBlank(" \t\n") // true}
   *   <li>{@code StrUtil.isBlank("abc") // false}
   * </ul>
   *
   * <p>注意：该方法与 {@link #isEmpty(CharSequence)} 的区别是： 该方法会校验空白字符，且性能相对于 {@link
   * #isEmpty(CharSequence)} 略慢。 <br>
   *
   * <p>建议：
   *
   * <ul>
   *   <li>该方法建议仅对于客户端（或第三方接口）传入的参数使用该方法。
   * </ul>
   *
   * @param str 被检测的字符串
   * @return 若为空白，则返回 true
   * @see #isEmpty(CharSequence)
   */
  public static boolean isBlank(CharSequence str) {
    final int length;
    if ((str == null) || ((length = str.length()) == 0)) {
      return true;
    }

    for (int i = 0; i < length; i++) {
      // 只要有一个非空字符即为非空字符串
      if (!CharUtil.isBlankChar(str.charAt(i))) {
        return false;
      }
    }

    return true;
  }

  /**
   * 字符串是否为空，空的定义如下：
   *
   * <ol>
   *   <li>{@code null}
   *   <li>空字符串：{@code ""}
   * </ol>
   *
   * <p>例：
   *
   * <ul>
   *   <li>{@code StrUtil.isEmpty(null) // true}
   *   <li>{@code StrUtil.isEmpty("") // true}
   *   <li>{@code StrUtil.isEmpty(" \t\n") // false}
   *   <li>{@code StrUtil.isEmpty("abc") // false}
   * </ul>
   *
   * <p>注意：该方法与 {@link #isBlank(CharSequence)} 的区别是：该方法不校验空白字符。
   *
   * <p>建议：
   *
   * <ul>
   *   <li>该方法建议用于工具类或任何可以预期的方法参数的校验中。
   * </ul>
   *
   * @param str 被检测的字符串
   * @return 是否为空
   */
  public static boolean isEmpty(CharSequence str) {
    return str == null || str.length() == 0;
  }

  /**
   * 大写首字母<br>
   * 例如：str = name, return Name
   *
   * @param str 字符串
   * @return 字符串
   */
  public static String upperFirst(CharSequence str) {
    if (null == str) {
      return null;
    }
    if (str.length() > 0) {
      char firstChar = str.charAt(0);
      if (Character.isLowerCase(firstChar)) {
        return Character.toUpperCase(firstChar) + subSuf(str, 1);
      }
    }
    return str.toString();
  }

  /**
   * 切割指定位置之后部分的字符串
   *
   * @param string 字符串
   * @param fromIndex 切割开始的位置（包括）
   * @return 切割后后剩余的后半部分字符串
   */
  public static String subSuf(CharSequence string, int fromIndex) {
    if (isEmpty(string)) {
      return null;
    }
    return sub(string, fromIndex, string.length());
  }

  /**
   * 改进JDK subString<br>
   * index从0开始计算，最后一个字符为-1<br>
   * 如果from和to位置一样，返回 "" <br>
   * 如果from或to为负数，则按照length从后向前数位置，如果绝对值大于字符串长度，则from归到0，to归到length<br>
   * 如果经过修正的index中from大于to，则互换from和to example: <br>
   * abcdefgh 2 3 =》 c <br>
   * abcdefgh 2 -3 =》 cde <br>
   *
   * @param str String
   * @param fromIndexInclude 开始的index（包括）
   * @param toIndexExclude 结束的index（不包括）
   * @return 字串
   */
  public static String sub(CharSequence str, int fromIndexInclude, int toIndexExclude) {
    if (isEmpty(str)) {
      return str(str);
    }
    int len = str.length();

    if (fromIndexInclude < 0) {
      fromIndexInclude = len + fromIndexInclude;
      if (fromIndexInclude < 0) {
        fromIndexInclude = 0;
      }
    } else if (fromIndexInclude > len) {
      fromIndexInclude = len;
    }

    if (toIndexExclude < 0) {
      toIndexExclude = len + toIndexExclude;
      if (toIndexExclude < 0) {
        toIndexExclude = len;
      }
    } else if (toIndexExclude > len) {
      toIndexExclude = len;
    }

    if (toIndexExclude < fromIndexInclude) {
      int tmp = fromIndexInclude;
      fromIndexInclude = toIndexExclude;
      toIndexExclude = tmp;
    }

    if (fromIndexInclude == toIndexExclude) {
      return EMPTY;
    }

    return str.toString().substring(fromIndexInclude, toIndexExclude);
  }

  /**
   * {@link CharSequence} 转为字符串，null安全
   *
   * @param cs {@link CharSequence}
   * @return 字符串
   */
  public static String str(CharSequence cs) {
    return null == cs ? null : cs.toString();
  }

  /**
   * 比较两个字符串（大小写敏感）。
   *
   * <pre>
   * equals(null, null)   = true
   * equals(null, &quot;abc&quot;)  = false
   * equals(&quot;abc&quot;, null)  = false
   * equals(&quot;abc&quot;, &quot;abc&quot;) = true
   * equals(&quot;abc&quot;, &quot;ABC&quot;) = false
   * </pre>
   *
   * @param str1 要比较的字符串1
   * @param str2 要比较的字符串2
   * @return 如果两个字符串相同，或者都是{@code null}，则返回{@code true}
   */
  public static boolean equals(CharSequence str1, CharSequence str2) {
    return equals(str1, str2, false);
  }

  /**
   * 比较两个字符串是否相等，规则如下
   *
   * <ul>
   *   <li>str1和str2都为{@code null}
   *   <li>忽略大小写使用{@link String#equalsIgnoreCase(String)}判断相等
   *   <li>不忽略大小写使用{@link String#contentEquals(CharSequence)}判断相等
   * </ul>
   *
   * @param str1 要比较的字符串1
   * @param str2 要比较的字符串2
   * @param ignoreCase 是否忽略大小写
   * @return 如果两个字符串相同，或者都是{@code null}，则返回{@code true}
   * @since 3.2.0
   */
  public static boolean equals(CharSequence str1, CharSequence str2, boolean ignoreCase) {
    if (null == str1) {
      // 只有两个都为null才判断相等
      return str2 == null;
    }
    if (null == str2) {
      // 字符串2空，字符串1非空，直接false
      return false;
    }

    if (ignoreCase) {
      return str1.toString().equalsIgnoreCase(str2.toString());
    } else {
      return str1.toString().contentEquals(str2);
    }
  }

  // ------------------------------------------------------------------------ strip

  /**
   * 去除两边的指定字符串
   *
   * @param str 被处理的字符串
   * @param prefixOrSuffix 前缀或后缀
   * @return 处理后的字符串
   * @since 3.1.2
   */
  public static String strip(CharSequence str, CharSequence prefixOrSuffix) {
    if (equals(str, prefixOrSuffix)) {
      // 对于去除相同字符的情况单独处理
      return EMPTY;
    }
    return strip(str, prefixOrSuffix, prefixOrSuffix);
  }

  /**
   * 去除两边的指定字符串
   *
   * @param str 被处理的字符串
   * @param prefix 前缀
   * @param suffix 后缀
   * @return 处理后的字符串
   * @since 3.1.2
   */
  public static String strip(CharSequence str, CharSequence prefix, CharSequence suffix) {
    if (isEmpty(str)) {
      return str(str);
    }

    int from = 0;
    int to = str.length();

    String str2 = str.toString();
    if (startWith(str2, prefix)) {
      from = prefix.length();
    }
    if (endWith(str2, suffix)) {
      to -= suffix.length();
    }

    return str2.substring(Math.min(from, to), Math.max(from, to));
  }

  /**
   * 去除两边的指定字符串，忽略大小写
   *
   * @param str 被处理的字符串
   * @param prefixOrSuffix 前缀或后缀
   * @return 处理后的字符串
   * @since 3.1.2
   */
  public static String stripIgnoreCase(CharSequence str, CharSequence prefixOrSuffix) {
    return stripIgnoreCase(str, prefixOrSuffix, prefixOrSuffix);
  }

  /**
   * 去除两边的指定字符串，忽略大小写
   *
   * @param str 被处理的字符串
   * @param prefix 前缀
   * @param suffix 后缀
   * @return 处理后的字符串
   * @since 3.1.2
   */
  public static String stripIgnoreCase(CharSequence str, CharSequence prefix, CharSequence suffix) {
    if (isEmpty(str)) {
      return str(str);
    }
    int from = 0;
    int to = str.length();

    String str2 = str.toString();
    if (startWithIgnoreCase(str2, prefix)) {
      from = prefix.length();
    }
    if (endWithIgnoreCase(str2, suffix)) {
      to -= suffix.length();
    }
    return str2.substring(from, to);
  }

  /**
   * 是否以指定字符串开头
   *
   * @param str 被监测字符串
   * @param prefix 开头字符串
   * @return 是否以指定字符串开头
   */
  public static boolean startWith(CharSequence str, CharSequence prefix) {
    return startWith(str, prefix, false);
  }

  /**
   * 是否以指定字符串开头<br>
   * 如果给定的字符串和开头字符串都为null则返回true，否则任意一个值为null返回false
   *
   * @param str 被监测字符串
   * @param prefix 开头字符串
   * @param ignoreCase 是否忽略大小写
   * @return 是否以指定字符串开头
   * @since 5.4.3
   */
  public static boolean startWith(CharSequence str, CharSequence prefix, boolean ignoreCase) {
    return startWith(str, prefix, ignoreCase, false);
  }

  /**
   * 是否以指定字符串开头<br>
   * 如果给定的字符串和开头字符串都为null则返回true，否则任意一个值为null返回false<br>
   *
   * <pre>
   *     CharSequenceUtil.startWith("123", "123", false, true);   -- false
   *     CharSequenceUtil.startWith("ABCDEF", "abc", true, true); -- true
   *     CharSequenceUtil.startWith("abc", "abc", true, true);    -- false
   * </pre>
   *
   * @param str 被监测字符串
   * @param prefix 开头字符串
   * @param ignoreCase 是否忽略大小写
   * @param ignoreEquals 是否忽略字符串相等的情况
   * @return 是否以指定字符串开头
   * @since 5.4.3
   */
  public static boolean startWith(
      CharSequence str, CharSequence prefix, boolean ignoreCase, boolean ignoreEquals) {
    if (null == str || null == prefix) {
      if (ignoreEquals) {
        return false;
      }
      return null == str && null == prefix;
    }

    boolean isStartWith =
        str.toString().regionMatches(ignoreCase, 0, prefix.toString(), 0, prefix.length());

    if (isStartWith) {
      return (false == ignoreEquals) || (false == equals(str, prefix, ignoreCase));
    }
    return false;
  }

  /**
   * 是否以指定字符串开头，忽略大小写
   *
   * @param str 被监测字符串
   * @param prefix 开头字符串
   * @return 是否以指定字符串开头
   */
  public static boolean startWithIgnoreCase(CharSequence str, CharSequence prefix) {
    return startWith(str, prefix, true);
  }

  /**
   * 是否以指定字符串结尾，忽略大小写
   *
   * @param str 被监测字符串
   * @param suffix 结尾字符串
   * @return 是否以指定字符串结尾
   */
  public static boolean endWithIgnoreCase(CharSequence str, CharSequence suffix) {
    return endWith(str, suffix, true);
  }

  /**
   * 是否以指定字符串结尾
   *
   * @param str 被监测字符串
   * @param suffix 结尾字符串
   * @return 是否以指定字符串结尾
   */
  public static boolean endWith(CharSequence str, CharSequence suffix) {
    return endWith(str, suffix, false);
  }

  /**
   * 是否以指定字符串结尾<br>
   * 如果给定的字符串和开头字符串都为null则返回true，否则任意一个值为null返回false
   *
   * @param str 被监测字符串
   * @param suffix 结尾字符串
   * @param ignoreCase 是否忽略大小写
   * @return 是否以指定字符串结尾
   */
  public static boolean endWith(CharSequence str, CharSequence suffix, boolean ignoreCase) {
    return endWith(str, suffix, ignoreCase, false);
  }

  /**
   * 是否以指定字符串结尾<br>
   * 如果给定的字符串和开头字符串都为null则返回true，否则任意一个值为null返回false
   *
   * @param str 被监测字符串
   * @param suffix 结尾字符串
   * @param ignoreCase 是否忽略大小写
   * @param ignoreEquals 是否忽略字符串相等的情况
   * @return 是否以指定字符串结尾
   * @since 5.8.0
   */
  public static boolean endWith(
      CharSequence str, CharSequence suffix, boolean ignoreCase, boolean ignoreEquals) {
    if (null == str || null == suffix) {
      if (ignoreEquals) {
        return false;
      }
      return null == str && null == suffix;
    }

    final int strOffset = str.length() - suffix.length();
    boolean isEndWith =
        str.toString().regionMatches(ignoreCase, strOffset, suffix.toString(), 0, suffix.length());

    if (isEndWith) {
      return (false == ignoreEquals) || (false == equals(str, suffix, ignoreCase));
    }
    return false;
  }
}
