package org.elsfs.tool.core.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * 集合工具类
 *
 * @author zeng
 * @since 0.0.1
 */
public class CollectionUtils {

  /**
   * 判断集合是否为空
   *
   * @param collection 需要判断的集合
   * @return 为空返回true，否则返回 false
   */
  public static boolean isEmpty(Collection<?> collection) {
    return (collection == null || collection.isEmpty());
  }

  /**
   * 判断 map 是否为空
   *
   * @param map 要检查的map
   * @return 给定的Map是否为空
   */
  public static boolean isEmpty(Map<?, ?> map) {
    return (map == null || map.isEmpty());
  }

  /**
   * 非空判断
   *
   * @param collection 要检查的 collection
   * @return 如果不为空则返回true，为空返回 false
   */
  public static boolean isNotEmpty(Collection<?> collection) {
    return !isEmpty(collection);
  }

  /**
   * 非空判断
   *
   * @param map 需要验证的map
   * @return 非空为true 否则为false
   */
  public static boolean isNotEmpty(Map<?, ?> map) {
    return !isEmpty(map);
  }

  /**
   * 获取集合的第一个非空元素
   *
   * @param <T> 集合元素类型
   * @param iterable {@link Iterable}
   * @return 第一个元素
   */
  public static <T> T getFirstNoneNull(Iterable<T> iterable) {
    if (null == iterable) {
      return null;
    }
    return getFirstNoneNull(iterable.iterator());
  }

  /**
   * 获取集合的第一个非空元素
   *
   * @param <T> 集合元素类型
   * @param iterator {@link Iterator}
   * @return 第一个非空元素，null表示未找到
   */
  public static <T> T getFirstNoneNull(Iterator<T> iterator) {
    return firstMatch(iterator, Objects::nonNull);
  }

  /**
   * 返回{@link Iterator}中第一个匹配规则的值
   *
   * @param <T> 数组元素类型
   * @param iterator {@link Iterator}
   * @param matcher 匹配接口，实现此接口自定义匹配规则
   * @return 匹配元素，如果不存在匹配元素或{@link Iterator}为空，返回 {@code null}
   */
  public static <T> T firstMatch(Iterator<T> iterator, Function<T, Boolean> matcher) {
    Objects.requireNonNull(matcher, "Matcher must be not null !");
    if (null != iterator) {
      while (iterator.hasNext()) {
        final T next = iterator.next();
        if (matcher.apply(next)) {
          return next;
        }
      }
    }
    return null;
  }
}
