/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Factory for creating {@link DayDate} objects.
 * 
 * @author Andreas Enblom
 */
public interface DayDateFactory {

    /**
     * @return The current date in the default timezone.
     */
    DayDate now();

    /**
     * 
     * @return The current date in the UTC timezone.
     */
    DayDate utc();

    /**
     * Returns a default date value: 1970-01-01 (the UNIX epoch date). This is
     * useful instead of <code>null</code> values.
     * 
     * @return The default date: 1970-01-01 00:00:00.000.
     */
    DayDate getDefault();

    /**
     * Creates a new {@link DayDate} object with date of the given string
     * serialization, as returned by {@link DayDate#serialize()}
     * 
     * @param date The string serialization of the date, on the format
     *             <code>YYYYMMDD</code>.
     * @return The deserialized day date object.
     * @throws IllegalArgumentException If the given string serialization does
     *                                  not represent a correct date.
     */
    DayDate deserialize(String date) throws IllegalArgumentException;

    /**
     * Creates a new {@link DayDate} object, if possible, from the given
     * parameters. If the given parameters do not represent a date,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @return The date object, or <code>null</code> if the provided parameters
     *         do not represent a correct date.
     */
    DayDate parse(String year, String month, String day);

    /**
     * Creates a new {@link DayDate} object, if possible, from the given
     * parameters. If the given parameters do not represent a date,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @return The date object, or <code>null</code> if the provided parameters
     *         do not represent a correct date.
     */
    DayDate parse(int year, int month, int day);

    /**
     * Creates a new {@link DayDate} object, if possible, from the given
     * string representation. If the given representation do not represent a
     * date, <code>null</code> is returned.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @return The date object, or <code>null</code> if the provided string
     *         representation does not represent a correct date.
     */
    DayDate parse(String date);

}
