/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link DayDate}.
 * 
 * @author Andreas Enblom
 */
class DayDateImpl implements DayDate, java.io.Serializable {

    static final int MAX_OFFSET = offset((short) 9999, (byte) 12, (byte) 31);

    final short year;   // 1000 - 9999
    final byte  month;  // 1 - 12
    final byte  day;    // 1 - 31

    /**
     * Creates a new instance with the given year, month, day. No verification
     * that the numbers are in the correct range, or that the date is valid, is
     * done.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day of month in the range 1-28, 1-29, 1-30 or 1-31
     *            depending on month and year.
     */
    DayDateImpl(short year, byte month, byte day) {
        this.year = year;
        this.month = month;
        this.day = day;
    }

    @Override
    public DayOfWeek getDayOfWeek() {
        return getDayOfWeek(year, month, day);
    }

    @Override
    public int year() {
        return year;
    }

    @Override
    public Month month() {
        return Month.fromNum(month);
    }

    @Override
    public int day() {
        return day;
    }

    @Override
    public DayDate plusYears(int offset) {
        DayDate date = new DayDateImpl(
                addYearGetNewYear(year, offset),
                addYearGetNewMonth(year, month, offset),
                addYearGetNewDay(year, month, day, offset));
        return date;
    }

    @Override
    public DayDate plusMonths(int offset) {
        DayDate date = new DayDateImpl(
                addMonthGetNewYear(year, month, offset),
                addMonthGetNewMonth(month, offset),
                addMonthGetNewDay(year, month, day, offset));
        return date;
    }

    @Override
    public DayDate plusDays(int offset) {
        return plusDays(year, month, day, offset);
    }

    static DayDateImpl plusDays(short year, byte month, byte day, int offset) {
        int newOffset = offset(year, month, day) + offset;
        if (newOffset < 0 || newOffset > DayDateImpl.MAX_OFFSET) {
            throw new TimeOutOfRangeException("The given offset results in a date that is out of range. Correct range is 1000-01-01--9999-12-31");
        }
        DayDateImpl newDate = fromOffset(newOffset);
        return newDate;
    }

    @Override
    public boolean isLaterYearThan(DayDate date) {
        return year > date.year();
    }

    @Override
    public boolean isSameYearAs(DayDate date) {
        return year == date.year();
    }

    @Override
    public boolean isLaterMonthThan(DayDate date) {
        return
        (year > date.year()) ||
        (year == date.year() && month > date.month().toNum());
    }

    @Override
    public boolean isSameMonthAs(DayDate date) {
        return year == date.year() && month() == date.month();
    }

    @Override
    public int compareTo(DayDate other) {
        return this.toInt() - other.toInt();
    }

    @Override
    public DayDateFormatter iso() {
        return new DayDateFormatterIsoImpl(this);
    }

    @Override
    public DayDateFormatter eur() {
        return new DayDateFormatterEurImpl(this);
    }

    @Override
    public DayDateFormatter us() {
        return new DayDateFormatterUsImpl(this);
    }

    @Override
    public String serialize() {
        return iso().formatCompactDate();
    }

    @Override
    public String toString() {
        return iso().formatDate();
    }

    @Override
    public int toInt() {
        return year * 10000 + month * 100 + day;
    }

    @Override
    public int hashCode() {
        return toInt();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DayDate)) {
            return false;
        }
        return this.toInt() == ((DayDate) obj).toInt();
    }

    /**
     * Calculcates the offset in days of the given date, from Jan 1, year 1000.
     * Example: The offset for 1000-01-01 is 0, the offset for 1000-01-02 is 1.
     * <p>
     * Note: No range check is made of the provided parameters, so make sure the
     * parameters are within the correct ranges.
     * 
     * @param year The year of the date, in the range 1000-9999.
     * @param month The month of the date, in the range 1-12.
     * @param day The day of the date, in the range 1-28, 1-29, 1-30 or 1-31.
     * @return The offset in days from 1000-01-01.
     */
    static int offset(short year, byte month, byte day) {
        /*
         * 1. Start at Jan 1st of year 1000 - offset 0.
         */
        int offset = 0;

        /*
         * 2. Jump forward to Jan 1st of the given year, keeping precise track
         *    of leap years.
         */
        offset += 365 * (year - 1000);
        offset += 1 * (year - 1001 + 4) / 4;     // Add one day for every fourth year after 1000 (i.e. for 1001, 1005, 1009 etc)...
        offset -= 1 * (year - 1001 + 100) / 100; // ...but not for years divisible by 100 (i.e. for 1001, 1101, 1201 etc)...
        offset += 1 * (year - 1201 + 400) / 400; // ...except years divisible by 400 (i.e. for 1201, 1601, 2001 etc).

        /*
         * 3. Jump forward to the given day of the given month.
         */
        switch(month) { // Fallthrough in all cases
        case 12: offset += 30;
        case 11: offset += 31;
        case 10: offset += 30;
        case 9:  offset += 31;
        case 8:  offset += 31;
        case 7:  offset += 30;
        case 6:  offset += 31;
        case 5:  offset += 30;
        case 4:  offset += 31;
        case 3:  offset += isLeapYear(year) ? 29 : 28;
        case 2:  offset += 31;
        }

        offset += day - 1;

        /*
         * 4. We're done!
         */
        return offset;
    }

    /**
     * Given an offset in days from 1000-01-01, calculates the date with that
     * offset. The is the inverse of {@link #offset(short, byte, byte)}.
     * <p>
     * Note: No range check of the offset is made, so make sure that the given
     * offset is within range.
     * 
     * @param offset The offset in whole days since 1000-01-01, starting with 0
     *               for that date. This should always be in the range 0 -
     *               {@link #MAX_OFFSET}.
     * @return The date with the given offset.
     */
    static DayDateImpl fromOffset(int offset) {

        // Note: This is the direct inverse of the function defined by the
        // method offset(short, byte, byte), and it might be useful to study
        // that implementation first. The below calculations are complicated by
        // leap years. Remember that every fourth year is a leap year (i.e.
        // years 4, 8, 12 etc), except years divisible by 100 (i.e. 100, 200
        // etc), unless the year is divisible by 400 (so 400, 800, 1200 are leap
        // years).

        /*
         * 1. Shift the offset to be an offset from Jan 1, year 1 instead of
         *    from Jan 1, year 1000. This is convenient since the 400-periodic
         *    cycle of leap years starts at year 1.
         */
        int offset0 = offset + 999*365 + 999/4 - 999/100 + 999/400;

        /*
         * 2. Now calculate the year with that offset. This calculcation is
         *    based on the fact that the number of days per year is on average
         *    365 + 1/4 - 1/100 + 1/400 = 146097 / 400. Experimentally, we know
         *    that dividing (integer division) the offset by this number (and
         *    adding 1, since the offset is from year 1, not year 0), we get the
         *    correct year, or on occasion one less than the correct year.
         */
        short year = (short) ((offset0 * 400) / 146097 + 1);
        if ((year*365 + year/4 - year/100 + year/400) <= offset0) {
            year++;
        }
        int remainder = offset0 - ((year-1)*365 + (year-1)/4 - (year-1)/100 + (year-1)/400);

        /*
         * 3. Find the month and day of the calculated year.
         */
        byte month;
        byte day;

        if (remainder < 31) {
            // January
            month = 1;
            day = (byte) (remainder + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28)) {
            // February
            month = 2;
            day = (byte) (remainder - 31 + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31) {
            // March
            month = 3;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28)) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30) {
            // April
            month = 4;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31) {
            // May
            month = 5;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30) {
            // June
            month = 6;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31) {
            // July
            month = 7;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31) {
            // August
            month = 8;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30) {
            // September
            month = 9;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31) {
            // October
            month = 10;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30) + 1);

        } else if (remainder < 31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30) {
            // November
            month = 11;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31) + 1);

        } else {
            // December
            month = 12;
            day = (byte) (remainder - (31 + (isLeapYear(year) ? 29 : 28) + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30) + 1);
        }

        /*
         * 4. We're done!
         */
        return new DayDateImpl(year, month, day);
    }

    static DayOfWeek getDayOfWeek(short year, byte month, byte day) {
        int offset = offset(year, month, day);
        switch(offset % 7) {
        case 0:  return DayOfWeek.WEDNESDAY; // Jan 1st of year 1000 was a Wednesday
        case 1:  return DayOfWeek.THURSDAY;
        case 2:  return DayOfWeek.FRIDAY;
        case 3:  return DayOfWeek.SATURDAY;
        case 4:  return DayOfWeek.SUNDAY;
        case 5:  return DayOfWeek.MONDAY;
        case 6:  return DayOfWeek.TUESDAY;
        default: throw new IllegalStateException("Unknown day of week: " + (offset % 7) + ". This should never happen!");
        }
    }

    static void checkYearRange(int year) throws TimeOutOfRangeException {
        if (year < 1000 || year > 9999) {
            throw new TimeOutOfRangeException("The year has to be within the range 1000-9999");
        }
    }

    static short addYearGetNewYear(short year, int offset) throws TimeOutOfRangeException {
        int newYear = year + offset;
        checkYearRange(newYear);
        return (short) newYear;
    }

    static byte addYearGetNewMonth(short year, byte month, int offset) {
        return month;
    }

    static byte addYearGetNewDay(short year, byte month, byte day, int offset) {
        short newYear = addYearGetNewYear(year, offset);
        byte maxDay = getMaxDay(newYear, month);
        return day > maxDay ? maxDay : day;
    }

    static short addMonthGetNewYear(short year, byte month, int offset) {
        int yearOffset = offset / 12;
        int newMonth = month + (offset % 12);
        if (newMonth < 1) {
            yearOffset--;
        } else if (newMonth > 12) {
            yearOffset++;
        }
        return addYearGetNewYear(year, yearOffset);
    }

    static byte addMonthGetNewMonth(byte month, int offset) {
        // Complicated expression to handle negative offsets
        return (byte) (((((month + offset - 1) % 12) + 12) % 12) + 1);
    }

    static byte addMonthGetNewDay(short year, byte month, byte day, int offset) {
        short newYear = addMonthGetNewYear(year, month, offset);
        byte newMonth = addMonthGetNewMonth(month, offset);
        byte maxDay   = getMaxDay(newYear, newMonth);
        return day > maxDay ? maxDay : day;
    }

    static byte getMaxDay(short year, byte month) {
        switch (month) {
        case  1: return 31;
        case  2: return (byte) (isLeapYear(year) ? 29 : 28);
        case  3: return 31;
        case  4: return 30;
        case  5: return 31;
        case  6: return 30;
        case  7: return 31;
        case  8: return 31;
        case  9: return 30;
        case 10: return 31;
        case 11: return 30;
        case 12: return 31;
        default: throw new IllegalStateException("Unknown month number: " + month);
        }
    }

    static boolean isLeapYear(short year) {
        if (year % 400 == 0) {
            return true;
        } else if (year % 100 == 0) {
            return false;
        } else if (year % 4 == 0) {
            return true;
        } else {
            return false;
        }
    }

}
