/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * A complete time with date and time of day, and a lot of utilities.
 * <p>
 * The time is relative to the local timezone, and contains no timezone
 * information. It is simply a "local time".
 * <p>
 * The framework is aware of leap years, but not of leap seconds nor daylight
 * saving time corrections.
 * <p>
 * The timestamp contains year, month, day, hour, minute, second and millisecond
 * information, and the valid range of times is from Jan 1, 1000, 00:00:00.000
 * to Dec 31, 9999, 23:59:59.999 (inclusive).
 * <p>
 * Time objects are always immutable - they cannot be changed once created.
 * <p>
 * The natural order of this class orders instances in chronological order,
 * and two instances are considered equal if and only if they coincide on all
 * parameters down to milliseconds.
 * <p>
 * <b>Example usage</b>
 * <table>
 * <tr valign="top">
 * <td>Create new instances:</td>
 * <td>
 * <code>Time.factory.now()</code><br>
 * <code>Time.factory.getDefault()</code><br>
 * <code>Time.factory.parse(...)</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Formatting:</td>
 * <td>
 * <code>Time.iso().formatXXX()</code><br>
 * <code>Time.eur().formatXXX()</code><br>
 * <code>Time.us().formatXXX()</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Serializing:</td>
 * <td>
 * <code>Time.serialize()</code><br>
 * <code>Time.factory.deserialize()</code>
 * </td>
 * </tr>
 * </table>
 * 
 * @author Andreas Enblom
 */
public interface Time extends Comparable<Time> {

    /**
     * A default implementation of a {@link TimeFactory}, for creating new
     * instances. The factory will create {@link java.io.Serializable
     * serializable} instances.
     */
    public static final TimeFactory factory = new TimeFactoryImpl();

    /**
     * @return The {@link DayDate date} represented by this time, a full date
     *         with year, month and day of month.
     */
    DayDate getDate();

    /**
     * @return The {@link TimeOfDay time of day} represented by this time.
     */
    TimeOfDay getTimeOfDay();

    /**
     * @return The year represented by this time in the range 1000-9999.
     */
    int year();

    /**
     * @return The {@link Month month} represented by this time.
     */
    Month month();

    /**
     * @return The day of month represented by this time in the range 1-28,
     *         1-29, 1-30 or 1-31, depending on what month and year it is.
     *         Leap-years are handled correctly.
     */
    int day();

    /**
     * @return The hour of the time of day represented by this time, in the
     *         range 0-23.
     */
    int hour();

    /**
     * @return The minute of the time of day represented by this time, in the
     *         range 0-59.
     */
    int minute();

    /**
     * @return The second of the time of day represented by this time, in the
     *         range 0-59.
     */
    int second();

    /**
     * @return The millisecond of the time of day represented by this time, in
     *         the range 0-999.
     */
    int millis();

    /**
     * Calculates the day of week of this time.
     * 
     * @return The day of week.
     */
    DayOfWeek getDayOfWeek();

    /**
     * Adds or subtracts the given number of years to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in years. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusYears(int offset);

    /**
     * Adds or subtracts the given number of months to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * <p>
     * If adding the given number of months would make the day of month over the
     * new month's maximum it will be reduced to the proper maximum. Hence,
     * adding three months to 2011-05-31 will give 2011-08-31, but adding four
     * will give 2011-09-30. And adding 13 months to 2010-01-30 will give the
     * date 2011-02-28.
     * 
     * @param offset The offset in months. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusMonths(int offset);

    /**
     * Adds or subtracts the given number of days to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in days. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusDays(int offset);

    /**
     * Adds or subtracts the given number of hours to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in hours. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusHours(int offset);

    /**
     * Adds or subtracts the given number of minutes to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in minutes. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusMinutes(long offset);

    /**
     * Adds or subtracts the given number of seconds to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in seconds. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusSeconds(long offset);

    /**
     * Adds or subtracts the given number of milliseconds to this time. If the
     * resulting time is out of range, a runtime exception is thrown.
     * 
     * @param offset The offset in millis. Can be positive, zero or negative.
     * @return The resulting time.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    Time plusMillis(long offset);

    /**
     * Determines if this is after (inclusive) the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is after the given time.
     */
    boolean isAfter(Time time);

    /**
     * Determines if this is after (inclusive) the given time of day.
     * 
     * @param timeOfDay The time of day to compare to.
     * @return Whether this is after the given time of day.
     */
    boolean isAfter(TimeOfDay timeOfDay);

    /**
     * Determines if this is after (inclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is after the given time of day.
     */
    boolean isAfter(int hours, int minutes);

    /**
     * Determines if this is after (inclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @param seconds The second part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is after the given time of day.
     */
    boolean isAfter(int hours, int minutes, int seconds);

    /**
     * Determines if this is before (exclusive) the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is before the given time.
     */
    boolean isBefore(Time time);

    /**
     * Determines if this is before (exclusive) the given time of day.
     * 
     * @param timeOfDay The time of day to compare to.
     * @return Whether this is before the given time of day.
     */
    boolean isBefore(TimeOfDay timeOfDay);

    /**
     * Determines if this is before (exclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is before the given time of day.
     */
    boolean isBefore(int hours, int minutes);

    /**
     * Determines if this is before (exclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @param seconds The second part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is before the given time of day.
     */
    boolean isBefore(int hours, int minutes, int seconds);

    /**
     * Determines whether this is in a later year than the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later year than the given time.
     */
    boolean isLaterYearThan(Time time);

    /**
     * Determines whether this is in the same year as the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same year as the given time.
     */
    boolean isSameYearAs(Time time);

    /**
     * Determines whether this is in a later month than the given time. A month
     * of a later year is always considered to be later, e.g. Jan, 2011 is a
     * later month than Nov, 2010.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later month than the given time.
     */
    boolean isLaterMonthThan(Time time);

    /**
     * Determines whether this is in the same month and the same year as the
     * given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same month as the given time.
     */
    boolean isSameMonthAs(Time time);

    /**
     * Determines whether this is in a later day than the given time. A day of a
     * later year is always considered to be later, e.g. Feb 5, 2011 is a later
     * day than Nov 10, 2010, and a later day than Jan 10, 2011.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later day than the given time.
     */
    boolean isLaterDayThan(Time time);

    /**
     * Determines whether this is in the same day (of the same year) as the
     * given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same day as the given time.
     */
    boolean isSameDayAs(Time time);

    /**
     * Determines whether this is in a later hour than the given time. A hour of
     * a later day is always considered to be later, e.g. 10:13 on Jan 19, 2011
     * is a later hour than 13:00 on Jan 18, 2011 and than 13:00 on Dec 20,
     * 2010.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later hour than the given time.
     */
    boolean isLaterHourThan(Time time);

    /**
     * Determines whether this is in the same hour (of the same day and year) as
     * the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same hour as the given time.
     */
    boolean isSameHourAs(Time time);

    /**
     * Determines whether this is in a later minute than the given time. A
     * minute of a later hour or day is always considered to be later, e.g.
     * 10:12 is a later minute than 9:13, and 10:13 on Jan 19 is a later minute
     * than 10:25 on Jan 18, 2011 and than 13:25 on Dec 20, 2010.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later minute than the given time.
     */
    boolean isLaterMinuteThan(Time time);

    /**
     * Determines whether this is in the same minute (of the same hour, day and
     * year) as the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same minute as the given time.
     */
    boolean isSameMinuteAs(Time time);

    /**
     * Determines whether this is in a later second than the given time. A
     * second of a later minute, hour or day is always considered to be later,
     * e.g. 10:13:15 is a later second than 10:12:25 and than 9:13:25. And
     * 10:13:15 on Jan 19 is a later second than 10:13:25 on Jan 18, 2011 and
     * than 13:20:25 on Dec 20, 2010.
     * 
     * @param time The time to compare to.
     * @return Whether this is in a later second than the given time.
     */
    boolean isLaterSecondThan(Time time);

    /**
     * Determines whether this is in the same second (of the same minute, hour,
     * day and year) as the given time.
     * 
     * @param time The time to compare to.
     * @return Whether this is in the same second as the given time.
     */
    boolean isSameSecondAs(Time time);

    @Override
    boolean equals(Object other);

    @Override
    int hashCode();

    /**
     * Provides a formatter for this time instance that formats timestamps
     * according to the ISO-8601 standard. For instance, this means that dates
     * are formatted as <code>YYYY-MM-DD</code> and <code>YYYYMMDD</code>.
     * <p>
     * See also {@link #eur()} and {@link #us()}.
     * 
     * @return A formatter for this time that uses the ISO-8601 standard.
     */
    TimeFormatter iso();

    /**
     * Provides a formatter for this time instance that formats timestamps
     * according to some de-facto European standard. For instance, this means
     * that dates are formatted as <code>DD.MM.YYYY</code> and
     * <code>DDMMYYYY</code>.
     * <p>
     * See also {@link #iso()} and {@link #us()}.
     * 
     * @return A formatter for this time that uses some de-facto European
     *         standard.
     */
    TimeFormatter eur();

    /**
     * Provides a formatter for this time instance that formats timestamps
     * according to the US standard. For instance, this means that dates are
     * formatted as <code>MM/DD/YYYY</code> and <code>MMDDYYYY</code>.
     * <p>
     * See also {@link #iso()} and {@link #eur()}.
     * 
     * @return A formatter for this time that uses the US standard.
     */
    TimeFormatter us();

    /**
     * @return The full timestamp on the format
     *         <code>YYYYMMDDhhmmssnnnn</code>, where <code>nnn</code> is the
     *         milliseconds.
     */
    String serialize();

    /**
     * Converts this time to a {@link java.util.Date} in the default timezone.
     * 
     * @return The converted date.
     */
    java.util.Date toJavaUtilDate();

    /**
     * Converts this time to a {@link java.util.Date} in the given timezone.
     * 
     * @param timeZone The time zone, or <code>null</code> to use the default
     *                 time zone.
     * @return The converted date.
     */
    java.util.Date toJavaUtilDate(java.util.TimeZone timeZone);

    /**
     * Overrides {@link Object#toString()}.
     * 
     * @return The timestamp on the format <code>YYYY-MM-DD hh:mm:ss.nnn</code>,
     *         where <code>nnn</code> is the milliseconds.
     */
    @Override
    String toString();

    /**
     * Creates an long representation of the timestamp, which when written in
     * base 10 is the string <code>YYYYMMDDhhmmssnnn</code>, where
     * <code>nnn</code> is the milliseconds. E.g, the timestamp 2011-03-02
     * 16:42:14.308 is represented as the long 20110302164214308. Note that the
     * maximum value is 99991231235959999, which is a valid long.
     * 
     * @return The long representation.
     */
    long toLong();

}
