/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link TimeFactory}, that produces {@link TimeImpl}
 * instances.
 * 
 * @author Andreas Enblom
 */
class TimeFactoryImpl implements TimeFactory {

    private static final java.util.TimeZone UTC = java.util.TimeZone.getTimeZone("UTC");
    private static final Time defaultTime = new TimeImpl((short) 1970, (byte) 1, (byte) 1,(byte) 0, (byte) 0, (byte) 0, (short) 0);

    @Override
    public Time now() {
        java.util.Calendar calendar = java.util.Calendar.getInstance();
        return convert(calendar);
    }

    @Override
    public Time utc() {
        java.util.Calendar calendar = java.util.Calendar.getInstance(UTC);
        return convert(calendar);
    }

    @Override
    public Time getDefault() {
        return defaultTime;
    }

    @Override
    public Time create(DayDate date, TimeOfDay timeOfDay) {
        if (date == null) {
            throw new IllegalArgumentException("The date cannot be null");
        }
        if (timeOfDay == null) {
            throw new IllegalArgumentException("The time of day cannot be null");
        }

        return new TimeImpl(
                (short) date.year(),
                (byte)  date.month().toNum(),
                (byte)  date.day(),
                (byte)  timeOfDay.hour(),
                (byte)  timeOfDay.minute(),
                (byte)  timeOfDay.second(),
                (short) timeOfDay.millis());
    }

    @Override
    public Time deserialize(String time) {
        if (time == null || time.length() != 17) {
            throw new IllegalArgumentException("The serialization string " + time + " is not correct");
        }

        // YYYY MM DD hh mm ss nnn
        // 0123 45 67 89 01 23 456
        String year   = time.substring(0, 4);
        String month  = time.substring(4, 6);
        String day    = time.substring(6, 8);
        String hour   = time.substring(8, 10);
        String minute = time.substring(10, 12);
        String second = time.substring(12, 14);
        String millis = time.substring(14, 17);

        Time deserialized = parse(year, month, day, hour, minute, second, millis);
        if (deserialized == null) {
            throw new IllegalArgumentException("The serialization string " + time + " is not correct");
        }

        return deserialized;
    }

    @Override
    public Time parse(String year, String month, String day, String hour, String minute, String second) {
        try {
            return parse(
                    Integer.parseInt(year),
                    Integer.parseInt(month),
                    Integer.parseInt(day),
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second),
                    0
            );
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(int year, int month, int day, int hour, int minute, int second) {
        return parse(year, month, day, hour, minute, second, 0);
    }

    @Override
    public Time parse(String year, String month, String day, String hour, String minute, String second, String millis) {
        try {
            return parse(
                    Integer.parseInt(year),
                    Integer.parseInt(month),
                    Integer.parseInt(day),
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second),
                    Integer.parseInt(millis));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(int year, int month, int day, int hour, int minute, int second, int millis) {
        if (year < 1000 || year > 9999) {
            return null;
        }
        if (month < 1 || month > 12) {
            return null;
        }
        if (day < 1 || day > DayDateImpl.getMaxDay((short) year, (byte) month)) {
            return null;
        }
        if (hour < 0 || hour > 23) {
            return null;
        }
        if (minute < 0 || minute > 59) {
            return null;
        }
        if (second < 0 || second > 59) {
            return null;
        }
        if (millis < 0 || millis > 999) {
            return null;
        }
        return new TimeImpl((short) year, (byte) month, (byte) day, (byte) hour, (byte) minute, (byte) second, (short) millis);
    }

    @Override
    public Time parse(String date, String hour, String minute, String second) {
        try {
            return parse(
                    date,
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(String date, int hour, int minute, int second) {
        return parse(date, hour, minute, second, 0);
    }

    @Override
    public Time parse(String date, String hour, String minute, String second, String millis) {
        try {
            return parse(
                    date,
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second),
                    Integer.parseInt(millis));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(String date, int hour, int minute, int second, int millis) {
        String year;
        String month;
        String day;

        if (date.length() == 10 && date.charAt(4) == '-' && date.charAt(7) == '-') {
            // Format: YYYY-MM-DD
            //         0123456789
            year  = date.substring(0, 4);
            month = date.substring(5, 7);
            day   = date.substring(8);

        } else if (date.length() == 8) {
            // Format: YYYYMMDD
            //         01234567
            year  = date.substring(0, 4);
            month = date.substring(4, 6);
            day   = date.substring(6);

        } else {
            // No supported format
            return null;
        }

        try {
            return parse(
                    Integer.parseInt(year),
                    Integer.parseInt(month),
                    Integer.parseInt(day),
                    hour,
                    minute,
                    second,
                    millis);
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(String date, TimeOfDay timeOfDay) {
        return parse(date, timeOfDay.hour(), timeOfDay.minute(), timeOfDay.second(), timeOfDay.millis());
    }

    @Override
    public Time parse(DayDate date, String hour, String minute, String second) {
        try {
            return parse(
                    date,
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(DayDate date, int hour, int minute, int second) {
        return parse(date, hour, minute, second, 0);
    }

    @Override
    public Time parse(DayDate date, String hour, String minute, String second, String millis) {
        try {
            return parse(
                    date,
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second),
                    Integer.parseInt(millis));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public Time parse(DayDate date, int hour, int minute, int second, int millis) {
        return parse(date.year(), date.month().toNum(), date.day(), hour, minute, second, millis);
    }

    @Override
    public Time parse(String time) {

        // Step 1: Split into date and time
        int space = time.indexOf(' ');
        if (space == -1) {
            return null;
        }

        String datePart = time.substring(0, space);
        String timePart = time.substring(space + 1);

        // Step 2: Parse date and check ranges
        String year;
        String month;
        String day;

        if (datePart.length() == 10 && datePart.charAt(4) == '-' && datePart.charAt(7) == '-') {
            // Format: YYYY-MM-DD
            //         0123456789
            year  = datePart.substring(0, 4);
            month = datePart.substring(5, 7);
            day   = datePart.substring(8);

        } else if (datePart.length() == 8) {
            // Format: YYYYMMDD
            //         01234567
            year  = datePart.substring(0, 4);
            month = datePart.substring(4, 6);
            day   = datePart.substring(6);

        } else {
            // No supported format
            return null;
        }

        // Step 3: Parse time and check ranges
        String hour;
        String minute;
        String second;
        String millis;

        if (timePart.length() == 6) {
            // Format: hhmmss
            //         012345
            hour   = timePart.substring(0, 2);
            minute = timePart.substring(2, 4);
            second = timePart.substring(4);
            millis = "0";

        } else if (timePart.length() == 9) {
            // Format: hhmmssnnn
            //         012345678
            hour   = timePart.substring(0, 2);
            minute = timePart.substring(2, 4);
            second = timePart.substring(4, 6);
            millis = timePart.substring(6);

        } else if (timePart.length() == 8 && timePart.charAt(2) == ':' && timePart.charAt(5) == ':') {
            // Format: hh:mm:ss
            //         01234567
            hour   = timePart.substring(0, 2);
            minute = timePart.substring(3, 5);
            second = timePart.substring(6);
            millis = "0";

        } else if (timePart.length() == 12 && timePart.charAt(2) == ':' && timePart.charAt(5) == ':' && timePart.charAt(8) == '.') {
            // Format: hh:mm:ss.nnn
            //         012345678901
            hour   = timePart.substring(0, 2);
            minute = timePart.substring(3, 5);
            second = timePart.substring(6, 8);
            millis = timePart.substring(9);

        } else {
            // No supported format
            return null;
        }

        return parse(year, month, day, hour, minute, second, millis);
    }

    @Override
    public Time fromJavaUtilDate(java.util.Date date) {
        java.util.Calendar calendar = java.util.Calendar.getInstance();
        calendar.setTime(date);
        return convert(calendar);
    }

    @Override
    public Time fromJavaUtilDate(java.util.Date date, java.util.TimeZone timeZone) {
        java.util.Calendar calendar = java.util.Calendar.getInstance(timeZone);
        calendar.setTime(date);
        return convert(calendar);
    }

    private static Time convert(java.util.Calendar calendar) {
        // Check the range of the year; the other field will not be out of range
        // as we trust java.util.Calendar on this point.
        int yearInt = calendar.get(java.util.Calendar.YEAR);
        DayDateImpl.checkYearRange(yearInt);

        short year = (short) yearInt;

        byte month;
        switch (calendar.get(java.util.Calendar.MONTH)) {
        case java.util.Calendar.JANUARY:
            month = 1;
            break;
        case java.util.Calendar.FEBRUARY:
            month = 2;
            break;
        case java.util.Calendar.MARCH:
            month = 3;
            break;
        case java.util.Calendar.APRIL:
            month = 4;
            break;
        case java.util.Calendar.MAY:
            month = 5;
            break;
        case java.util.Calendar.JUNE:
            month = 6;
            break;
        case java.util.Calendar.JULY:
            month = 7;
            break;
        case java.util.Calendar.AUGUST:
            month = 8;
            break;
        case java.util.Calendar.SEPTEMBER:
            month = 9;
            break;
        case java.util.Calendar.OCTOBER:
            month = 10;
            break;
        case java.util.Calendar.NOVEMBER:
            month = 11;
            break;
        case java.util.Calendar.DECEMBER:
            month = 12;
            break;
        default:
            throw new IllegalStateException("Unknown month from java.util.Calendar: " + calendar.get(java.util.Calendar.MONTH));
        }

        byte day = (byte) calendar.get(java.util.Calendar.DAY_OF_MONTH);

        byte hour = (byte) calendar.get(java.util.Calendar.HOUR_OF_DAY);
        byte minute = (byte) calendar.get(java.util.Calendar.MINUTE);
        byte second = (byte) calendar.get(java.util.Calendar.SECOND);
        short millis = (short) calendar.get(java.util.Calendar.MILLISECOND);

        return new TimeImpl(year, month, day, hour, minute, second, millis);
    }

}
