/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Represents a specific time of the day (hours, minutes, seconds, milliseconds)
 * in the range 00:00:00.000 - 23:59:59.000.
 * <p>
 * These time objects are always immutable - they cannot be changed once
 * created.
 * <p>
 * The natural order of this class orders times in chronological order, and two
 * instances are considered equal if they represent the same hour, minute,
 * second and millisecond.
 * <p>
 * <b>Example usage</b>
 * <table>
 * <tr valign="top">
 * <td>Create new instances:</td>
 * <td>
 * <code>TimeOfDay.factory.now()</code><br>
 * <code>TimeOfDay.factory.getDefault()</code><br>
 * <code>TimeOfDay.factory.parse(...)</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Formatting:</td>
 * <td>
 * <code>TimeOfDay.iso().formatXXX()</code><br>
 * <code>TimeOfDay.eur().formatXXX()</code><br>
 * <code>TimeOfDay.us().formatXXX()</code>
 * </td>
 * </tr>
 * <tr valign="top">
 * <td>Serializing</td>
 * <td>
 * <code>TimeOfDay.serialize()</code><br>
 * <code>TimeOfDay.factory.deserialize()</code>
 * </td>
 * </tr>
 * </table>
 * 
 * @author Andreas Enblom
 */
public interface TimeOfDay extends Comparable<TimeOfDay> {

    /**
     * A default implementation of a {@link TimeOfDayFactory}, for creating new
     * instances. The factory will create {@link java.io.Serializable
     * serializable} instances.
     */
    public static final TimeOfDayFactory factory = new TimeOfDayFactoryImpl();

    /**
     * @return The hour of the time of day, in the range 0-23.
     */
    int hour();

    /**
     * @return The minute of the time of day, in the range 0-59.
     */
    int minute();

    /**
     * @return The second of the time of day, in the range 0-59.
     */
    int second();

    /**
     * @return The millisecond of the time of day, in the range 0-999.
     */
    int millis();

    /**
     * Adds or subtracts the given number of hours to this time of day. If the
     * resulting time is out of range (for instance because it is past
     * midnight), a runtime exception is thrown.
     * 
     * @param offset The offset in hours. Can be positive, zero or negative.
     * @return The resulting time of day.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    TimeOfDay plusHours(int offset);

    /**
     * Adds or subtracts the given number of minutes to this time of day. If the
     * resulting time of day is out of range (for instance because it is past
     * midnight), a runtime exception is thrown.
     * 
     * @param offset The offset in minutes. Can be positive, zero or negative.
     * @return The resulting time of day.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    TimeOfDay plusMinutes(int offset);

    /**
     * Adds or subtracts the given number of seconds to this time of day. If the
     * resulting time of day is out of range (for instance because it is past
     * midnight), a runtime exception is thrown.
     * 
     * @param offset The offset in seconds. Can be positive, zero or negative.
     * @return The resulting time of day.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    TimeOfDay plusSeconds(int offset);

    /**
     * Adds or subtracts the given number of milliseconds to this time of day.
     * If the resulting time of day is out of range (for instance because it is
     * past midnight), a runtime exception is thrown.
     * 
     * @param offset The offset in millis. Can be positive, zero or negative.
     * @return The resulting time of day.
     * @throws TimeOutOfRangeException If the resulting time is out of range.
     */
    TimeOfDay plusMillis(int offset);

    /**
     * Determines if this is after (inclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is after the given time of day.
     */
    boolean isAfter(int hours, int minutes);

    /**
     * Determines if this is after (inclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @param seconds The second part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is after the given time of day.
     */
    boolean isAfter(int hours, int minutes, int seconds);

    /**
     * Determines if this is before (exclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is before the given time of day.
     */
    boolean isBefore(int hours, int minutes);

    /**
     * Determines if this is before (exclusive) the given time of day.
     * 
     * @param hours The hour part of the time of day to compare to, in the range
     *              0-23.
     * @param minutes The minute part of the time of day to compare to, in the
     *                range 0-59.
     * @param seconds The second part of the time of day to compare to, in the
     *                range 0-59.
     * @return Whether this is before the given time of day.
     */
    boolean isBefore(int hours, int minutes, int seconds);

    /**
     * Determines whether this is in a later hour than the given time of day.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in a later hour than the given time of day.
     */
    boolean isLaterHourThan(TimeOfDay time);

    /**
     * Determines whether this is in the same hour as the given time of day.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in the same hour as the given time of day.
     */
    boolean isSameHourAs(TimeOfDay time);

    /**
     * Determines whether this is in a later minute than the given time of day.
     * A minute of a later hour or day is always considered to be later, e.g.
     * 10:12 is a later minute than 9:13.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in a later minute than the given time of day.
     */
    boolean isLaterMinuteThan(TimeOfDay time);

    /**
     * Determines whether this is in the same minute (of the same hour) as the
     * given time of day.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in the same minute as the given time of day.
     */
    boolean isSameMinuteAs(TimeOfDay time);

    /**
     * Determines whether this is in a later second than the given time. A
     * second of a later minute or hour is always considered to be later, e.g.
     * 10:13:15 is a later second than 10:12:25 and than 9:14:25.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in a later second than the given time of day.
     */
    boolean isLaterSecondThan(TimeOfDay time);

    /**
     * Determines whether this is in the same second (of the same minute, and
     * hour) as the given time of day.
     * 
     * @param time The time of day to compare to.
     * @return Whether this is in the same second as the given time of day.
     */
    boolean isSameSecondAs(TimeOfDay time);

    @Override
    int hashCode();

    @Override
    boolean equals(Object other);

    /**
     * Provides a formatter for this time instance that formats time of day
     * according to the ISO-8601 standard.
     * <p>
     * See also {@link #eur()} and {@link #us()}. Incidentally, the three
     * formatting standards ISO, EUR and US coincide for times of day, but they
     * are still kept separate to follow the pattern of {@link Time} and
     * {@link DayDate} formatters.
     * 
     * @return A formatter for this time that uses the ISO-8601 standard.
     */
    TimeOfDayFormatter iso();

    /**
     * Provides a formatter for this time instance that formats times of day
     * according to some de-facto European standard.
     * <p>
     * See also {@link #iso()} and {@link #us()}. Incidentally, the three
     * formatting standards ISO, EUR and US coincide for times of day, but they
     * are still kept separate to follow the pattern of {@link Time} and
     * {@link DayDate} formatters.
     * 
     * @return A formatter for this time that uses some de-facto European
     *         standard.
     */
    TimeOfDayFormatter eur();

    /**
     * Provides a formatter for this time instance that formats times of day
     * according to the US standard.
     * <p>
     * See also {@link #iso()} and {@link #eur()}. Incidentally, the three
     * formatting standards ISO, EUR and US coincide for times of day, but they
     * are still kept separate to follow the pattern of {@link Time} and
     * {@link DayDate} formatters.
     * 
     * @return A formatter for this time that uses the US standard.
     */
    TimeOfDayFormatter us();

    /**
     * @return The time of day on the format <code>hhmmssnnn</code>, where
     *         <code>nnn</code> is the milliseconds.
     */
    String serialize();

    /**
     * Overrides {@link Object#toString()}.
     * 
     * @return The time of day on the format <code>hh:mm:ss.nnn</code>, where
     *         <code>nnn</code> is the milliseconds.
     */
    @Override
    String toString();

    /**
     * Creates an int representation of the time of day, which when written in
     * base 10 is the string <code>hhmmssnnn</code>, where <code>nnn</code> is
     * the milliseconds. E.g, the time 16:42:14.308 is represented as the
     * integer 164214308. Note that the maximum value is 235959999, which is a
     * valid int.
     * 
     * @return The int representation.
     */
    int toInt();

}
