/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Factory for creating {@link TimeOfDay} objects.
 * 
 * @author Andreas Enblom
 */
public interface TimeOfDayFactory {

    /**
     * @return The current time of day in the default timezone.
     */
    TimeOfDay now();

    /**
     * @return The current time of day in the UTC timezone.
     */
    TimeOfDay utc();

    /**
     * Returns a default time of day: 00:00:00.000 (midnight). This is useful
     * instead of <code>null</code> values.
     * 
     * @return The default time of day: 00:00:00.000.
     */
    TimeOfDay getDefault();

    /**
     * Creates a new {@link TimeOfDay} object with time of the given string
     * serialization, as returned by {@link TimeOfDay#serialize()}
     * 
     * @param time The string serialization of the time, on the format
     *             <code>hhmmssnnn</code>, where <code>nnn</code> is the
     *             milliseconds.
     * @return The deserialized time of day object.
     * @throws IllegalArgumentException If the given string serialization does
     *                                  not represent a correct time of day.
     */
    TimeOfDay deserialize(String time) throws IllegalArgumentException;

    /**
     * Creates a new {@link TimeOfDay} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time of
     * day, <code>null</code> is returned. This is the case even if the
     * parameters are totally wrong (i.e. not number, out of range, null etc
     * etc) - no exception is ever thrown.
     * 
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time of day object, or <code>null</code> if the provided
     *         parameters do not represent a correct time.
     */
    TimeOfDay parse(String hour, String minute, String second);

    /**
     * Creates a new {@link TimeOfDay} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time of
     * day, <code>null</code> is returned. This is the case even if the
     * parameters are totally wrong (i.e. out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time of day object, or <code>null</code> if the provided
     *         parameters do not represent a correct time.
     */
    TimeOfDay parse(int hour, int minute, int second);

    /**
     * Creates a new {@link TimeOfDay} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time of
     * day, <code>null</code> is returned. This is the case even if the
     * parameters are totally wrong (i.e. not number, out of range, null etc
     * etc) - no exception is ever thrown.
     * 
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time of day object, or <code>null</code> if the provided
     *         parameters do not represent a correct time.
     */
    TimeOfDay parse(String hour, String minute, String second, String millis);

    /**
     * Creates a new {@link TimeOfDay} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time of
     * day, <code>null</code> is returned. This is the case even if the
     * parameters are totally wrong (i.e. out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time of day object, or <code>null</code> if the provided
     *         parameters do not represent a correct time.
     */
    TimeOfDay parse(int hour, int minute, int second, int millis);

    /**
     * Creates a new {@link TimeOfDay} object, if possible, from the given
     * string representation. If the given representation do not represent a
     * correct time of day, <code>null</code> is returned.
     * <p>
     * If a millisecond value is not provided in the given string
     * representation, the millisecond is set to 0.
     * 
     * @param timeOfDay The time, represented as a string on one of the formats
     *                  <code>hhmmss</code>, <code>hhmmssnnn</code>,
     *                  <code>hh:mm:ss</code>, <code>hh:mm:ss.nnn</code>, where
     *                  <code>nnn</code> is the millisecond.
     * @return The time of day object, or <code>null</code> if the provided
     *         string representation does not represent a correct time of day.
     */
    TimeOfDay parse(String timeOfDay);

}
