/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link TimeOfDayFactory}, that produces
 * {@link TimeOfDayImpl} instances.
 * 
 * @author Andreas Enblom
 */
class TimeOfDayFactoryImpl implements TimeOfDayFactory {

    private static final java.util.TimeZone UTC = java.util.TimeZone.getTimeZone("UTC");
    private static final TimeOfDay defaultTimeOfDay = new TimeOfDayImpl((byte) 0, (byte) 0, (byte) 0, (short) 0);

    @Override
    public TimeOfDay now() {
        java.util.Calendar calendar = java.util.Calendar.getInstance();
        return convert(calendar);
    }

    @Override
    public TimeOfDay utc() {
        java.util.Calendar calendar = java.util.Calendar.getInstance(UTC);
        return convert(calendar);
    }

    @Override
    public TimeOfDay getDefault() {
        return defaultTimeOfDay;
    }

    @Override
    public TimeOfDay deserialize(String time) throws IllegalArgumentException {
        if (time == null || time.length() != 9) {
            throw new IllegalArgumentException("The serialization string " + time + " is not correct");
        }

        // hh mm ss nnn
        // 01 23 45 678
        String hour   = time.substring(0, 2);
        String minute = time.substring(2, 4);
        String second = time.substring(4, 6);
        String millis = time.substring(6, 9);

        TimeOfDay deserialized = parse(hour, minute, second, millis);
        if (deserialized == null) {
            throw new IllegalArgumentException("The serialization string " + time + " is not correct");
        }

        return deserialized;
    }

    @Override
    public TimeOfDay parse(String hour, String minute, String second) {
        try {
            return parse(
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public TimeOfDay parse(int hour, int minute, int second) {
        return parse(hour, minute, second, 0);
    }

    @Override
    public TimeOfDay parse(String hour, String minute, String second, String millis) {
        try {
            return parse(
                    Integer.parseInt(hour),
                    Integer.parseInt(minute),
                    Integer.parseInt(second),
                    Integer.parseInt(millis));
        } catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    public TimeOfDay parse(int hour, int minute, int second, int millis) {
        if (hour < 0 || hour > 23) {
            return null;
        }
        if (minute < 0 || minute > 59) {
            return null;
        }
        if (second < 0 || second > 59) {
            return null;
        }
        if (millis < 0 || millis > 999) {
            return null;
        }
        return new TimeOfDayImpl((byte) hour, (byte) minute, (byte) second, (short) millis);
    }

    @Override
    public TimeOfDay parse(String timeOfDay) {
        String hour;
        String minute;
        String second;
        String millis;

        if (timeOfDay.length() == 6) {
            // Format: hhmmss
            //         012345
            hour   = timeOfDay.substring(0, 2);
            minute = timeOfDay.substring(2, 4);
            second = timeOfDay.substring(4);
            millis = "0";

        } else if (timeOfDay.length() == 9) {
            // Format: hhmmssnnn
            //         012345678
            hour   = timeOfDay.substring(0, 2);
            minute = timeOfDay.substring(2, 4);
            second = timeOfDay.substring(4, 6);
            millis = timeOfDay.substring(6);

        } else if (timeOfDay.length() == 8 && timeOfDay.charAt(2) == ':' && timeOfDay.charAt(5) == ':') {
            // Format: hh:mm:ss
            //         01234567
            hour   = timeOfDay.substring(0, 2);
            minute = timeOfDay.substring(3, 5);
            second = timeOfDay.substring(6);
            millis = "0";

        } else if (timeOfDay.length() == 12 && timeOfDay.charAt(2) == ':' && timeOfDay.charAt(5) == ':' && timeOfDay.charAt(8) == '.') {
            // Format: hh:mm:ss.nnn
            //         012345678901
            hour   = timeOfDay.substring(0, 2);
            minute = timeOfDay.substring(3, 5);
            second = timeOfDay.substring(6, 8);
            millis = timeOfDay.substring(9);

        } else {
            // No supported format
            return null;
        }

        return parse(hour, minute, second, millis);
    }

    private static TimeOfDay convert(java.util.Calendar calendar) {
        // Don't check the range of the time fields - we trust
        // java.util.Calendar on this point.
        byte hour = (byte) calendar.get(java.util.Calendar.HOUR_OF_DAY);
        byte minute = (byte) calendar.get(java.util.Calendar.MINUTE);
        byte second = (byte) calendar.get(java.util.Calendar.SECOND);
        short millis = (short) calendar.get(java.util.Calendar.MILLISECOND);

        return new TimeOfDayImpl(hour, minute, second, millis);
    }

}
