/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Provides methods for formatting a {@link TimeOfDay} object as a
 * human-readable string.
 * <p>
 * The following notation will be used:
 * <table>
 * </tr><td><code>hh</code>:</td><td>The hour of the time written with two digits. A 24-hour scale is used, so this will be in the range 00-23.</td></tr>
 * </tr><td><code>mm</code>:</td><td>The minute of the time written with two digits; in the range 00-59.</td></tr>
 * </tr><td><code>ss</code>:</td><td>The second of the time written with two digits; in the range 00-59.</td></tr>
 * </tr><td><code>nnn</code>:</td><td>The millisecond of the time written with three digits; in the range 000-999.</td></tr>
 * </table>
 * <p>
 * There are three standard formatters: ISO, EUR and US, provided by
 * {@link Time#iso()}, {@link Time#eur()} and {@link Time#us()}, respectively.
 * The ISO formatter formats timestamps according to the ISO-8601 standard, the
 * EUR formatter formats timestamps according to some de-facto European
 * standard, and the US formatter formats timestamps according to US standards.
 * In the case of formatting a time of day, all these formatters coincide, but
 * they are still separated to follow the pattern of {@link TimeFormatter}
 * and {@link DayDateFormatter}.
 * 
 * @author Andreas Enblom
 */
public interface TimeOfDayFormatter {

    /**
     * Formats the time of day as a compact string, with no delimiters between
     * the different parts of the time. The formatting will include hour, minute
     * and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmmss</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmmss</code></td></tr>
     * <tr><td>US</td><td><code>hhmmss</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatCompactTime();

    /**
     * Formats the time of day as a compact string, with no delimiters between
     * the different parts of the time. The formatting will include hour,
     * minute, second and millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmmssnnn</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmmssnnn</code></td></tr>
     * <tr><td>US</td><td><code>hhmmssnnn</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatCompactLongTime();

    /**
     * Formats the time of day as a compact string, with no delimiters between
     * the different parts of the time. The formatting will include hour and
     * minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hhmm</code></td></tr>
     * <tr><td>EUR</td><td><code>hhmm</code></td></tr>
     * <tr><td>US</td><td><code>hhmm</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatCompactShortTime();

    /**
     * Formats the time of day as a string, with delimiters between the
     * different parts of this time. The formatting will include hour, minute
     * and second.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm:ss</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm:ss</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm:ss</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatTime();

    /**
     * Formats the time of day as a string, with delimiters between the
     * different parts of this time. The formatting will include hour, minute,
     * second and millisecond.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm:ss.nnn</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatLongTime();

    /**
     * Formats the time of day as a string, with delimiters between the
     * different parts of this time. The formatting will include hour and
     * minute.
     * <table>
     * <tr><th>Formatter</th><th>Format</th></tr>
     * <tr><td>ISO</td><td><code>hh:mm</code></td></tr>
     * <tr><td>EUR</td><td><code>hh:mm</code></td></tr>
     * <tr><td>US</td><td><code>hh:mm</code></td></tr>
     * </table>
     * 
     * @return The time of day, formatted as a string.
     */
    String formatShortTime();

}
