/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Default implementation of {@link TimeOfDay}.
 * 
 * @author Andreas Enblom
 */
class TimeOfDayImpl implements TimeOfDay, java.io.Serializable {

    final byte  hour;   // 0 - 23
    final byte  minute; // 0 - 59
    final byte  second; // 0 - 59
    final short millis; // 0 - 999

    /**
     * Creates a new instance with the given hour, minute, second and
     * millisecond. No verification that the numbers are in the correct range is
     * done.
     * 
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     */
    TimeOfDayImpl(byte hour, byte minute, byte second, short millis) {
        this.hour = hour;
        this.minute = minute;
        this.second = second;
        this.millis = millis;
    }

    @Override
    public int hour() {
        return hour;
    }

    @Override
    public int minute() {
        return minute;
    }

    @Override
    public int second() {
        return second;
    }

    @Override
    public int millis() {
        return millis;
    }

    @Override
    public TimeOfDay plusHours(int offset) {
        return new TimeOfDayImpl(
                addHoursGetNewHour(hour, offset),
                minute,
                second,
                millis);
    }

    @Override
    public TimeOfDay plusMinutes(int offset) {
        return new TimeOfDayImpl(
                addMinutesGetNewHour(hour, minute, offset),
                TimeImpl.addMinutesGetNewMinute(minute, offset),
                second,
                millis);
    }

    @Override
    public TimeOfDay plusSeconds(int offset) {
        return new TimeOfDayImpl(
                addSecondsGetNewHour(hour, minute, second, offset),
                addSecondsGetNewMinute(minute, second, offset),
                TimeImpl.addSecondsGetNewSecond(second, offset),
                millis);
    }

    @Override
    public TimeOfDay plusMillis(int offset) {
        return new TimeOfDayImpl(
                addMillisGetNewHour(hour, minute, second, millis, offset),
                addMillisGetNewMinute(minute, second, millis, offset),
                addMillisGetNewSecond(second, millis, offset),
                TimeImpl.addMillisGetNewMillis(millis, offset));
    }

    @Override
    public boolean isAfter(int hours, int minutes) {
        if (hour > hours) {
            return true;
        } else if (hour == hours && minute >= minutes) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isAfter(int hours, int minutes, int seconds) {
        if (hour > hours) {
            return true;
        } else if (hour == hours && minute > minutes) {
            return true;
        } else if (hour == hours && minute == minutes && second >= seconds) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isBefore(int hours, int minutes) {
        if (hour < hours) {
            return true;
        } else if (hour == hours && minute < minutes) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isBefore(int hours, int minutes, int seconds) {
        if (hour < hours) {
            return true;
        } else if (hour == hours && minute < minutes) {
            return true;
        } else if (hour == hours && minute == minutes && second < seconds) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isLaterHourThan(TimeOfDay time) {
        return hour > time.hour();
    }

    @Override
    public boolean isSameHourAs(TimeOfDay time) {
        return hour == time.hour();
    }

    @Override
    public boolean isLaterMinuteThan(TimeOfDay time) {
        return
        (hour > time.hour()) ||
        (hour == time.hour() && minute > time.minute());
    }

    @Override
    public boolean isSameMinuteAs(TimeOfDay time) {
        return hour == time.hour() && minute == time.minute();
    }

    @Override
    public boolean isLaterSecondThan(TimeOfDay time) {
        return
        (hour > time.hour()) ||
        (hour == time.hour() && minute > time.minute()) ||
        (hour == time.hour() && minute == time.minute() && second > time.second());
    }

    @Override
    public boolean isSameSecondAs(TimeOfDay time) {
        return hour == time.hour() && minute == time.minute() && second == time.second();
    }

    @Override
    public int compareTo(TimeOfDay other) {
        return this.toInt() - other.toInt();
    }

    @Override
    public TimeOfDayFormatter iso() {
        return new TimeOfDayFormatterImpl(this);
    }

    @Override
    public TimeOfDayFormatter eur() {
        return new TimeOfDayFormatterImpl(this);
    }

    @Override
    public TimeOfDayFormatter us() {
        return new TimeOfDayFormatterImpl(this);
    }

    @Override
    public String serialize() {
        return iso().formatCompactLongTime();
    }

    @Override
    public String toString() {
        return iso().formatLongTime();
    }

    @Override
    public int toInt() {
        return hour * 10000000 + minute * 100000 + second * 1000 + millis;
    }

    @Override
    public int hashCode() {
        return toInt();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TimeOfDay)) {
            return false;
        }
        return this.toInt() == ((TimeOfDay) obj).toInt();
    }

    private static void checkHourRange(int hour) {
        if (hour < 0 || hour > 23) {
            throw new TimeOutOfRangeException("The hour has to be within the range 0 - 23");
        }
    }

    private static byte addHoursGetNewHour(byte hour, int offset) {
        int newHour = hour + offset;
        checkHourRange(newHour);
        return (byte) newHour;
    }

    private static byte addMinutesGetNewHour(byte hour, byte minute, long offset) {
        int hourOffset = TimeImpl.addMinutesGetHourOffset(minute, offset);
        return addHoursGetNewHour(hour, hourOffset);
    }

    private static byte addSecondsGetNewHour(byte hour, byte minute, byte second, long offset) {
        long minuteOffset = TimeImpl.addSecondsGetMinuteOffset(second, offset);
        return addMinutesGetNewHour(hour, minute, minuteOffset);
    }

    private static byte addSecondsGetNewMinute(byte minute, byte second, long offset) {
        long minuteOffset = TimeImpl.addSecondsGetMinuteOffset(second, offset);
        return TimeImpl.addMinutesGetNewMinute(minute, minuteOffset);
    }

    private static byte addMillisGetNewHour(byte hour, byte minute, byte second, short millis, long offset) {
        long secondOffset = TimeImpl.addMillisGetSecondOffset(millis, offset);
        return addSecondsGetNewHour(hour, minute, second, secondOffset);
    }

    private static byte addMillisGetNewMinute(byte minute, byte second, short millis, long offset) {
        long secondOffset = TimeImpl.addMillisGetSecondOffset(millis, offset);
        return addSecondsGetNewMinute(minute, second, secondOffset);
    }

    private static byte addMillisGetNewSecond(byte second, short millis, long offset) {
        long secondOffset = TimeImpl.addMillisGetSecondOffset(millis, offset);
        return TimeImpl.addSecondsGetNewSecond(second, secondOffset);
    }

}
