/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Utilities for dealing with {@link Time} and similar objects.
 * 
 * @author Andreas Enblom
 */
public class TimeUtils {

    /**
     * Finds the minimum of two times.
     * 
     * @param t1 The first time.
     * @param t2 The last time
     * @return The minimum (i.e. earliest) of the two times.
     */
    public static Time min(Time t1, Time t2) {
        return t2.isBefore(t1) ? t2 : t1;
    }

    /**
     * Finds the maximum of two times.
     * 
     * @param t1 The first time.
     * @param t2 The last time
     * @return The maximum (i.e. latest) of the two times.
     */
    public static Time max(Time t1, Time t2) {
        return t2.isAfter(t1) ? t2 : t1;
    }

    /**
     * Finds the minimum of two dates.
     * 
     * @param d1 The first date.
     * @param d2 The last date
     * @return The minimum (i.e. earliest) of the two date.
     */
    public static DayDate min(DayDate d1, DayDate d2) {
        return d1.compareTo(d2) <= 0 ? d1 : d2;
    }

    /**
     * Finds the maximum of two dates.
     * 
     * @param d1 The first time.
     * @param d2 The last time
     * @return The maximum (i.e. latest) of the two dates.
     */
    public static DayDate max(DayDate d1, DayDate d2) {
        return d1.compareTo(d2) >= 0 ? d1 : d2;
    }

    /**
     * Finds the minimum of two times.
     * 
     * @param t1 The first time.
     * @param t2 The last time
     * @return The minimum (i.e. earliest) of the two times.
     */
    public static TimeOfDay min(TimeOfDay t1, TimeOfDay t2) {
        return t1.compareTo(t2) <= 0 ? t1 : t2;
    }

    /**
     * Finds the maximum of two times.
     * 
     * @param t1 The first time.
     * @param t2 The last time
     * @return The maximum (i.e. latest) of the two times.
     */
    public static TimeOfDay max(TimeOfDay t1, TimeOfDay t2) {
        return t1.compareTo(t2) >= 0 ? t1 : t2;
    }

    /**
     * Calculates the difference in whole days between the two given times. Only
     * the date part of the timestamp is considered, so the difference between
     * 2011-06-04 00:01 and 2011-06-03 23:59 is one day. The sign of this
     * difference assumes that the first argument is later than the second
     * argument, if not, the calculated difference will be negative. Two times
     * representing the same day will have zero difference.
     * 
     * @param later The first time to compare.
     * @param earlier The second time to compare.
     * @return The difference in whole days between the two given times. This
     *         is positive if the first argument represents a later day than
     *         the second argument, negative if the first argument represents an
     *         earlier day than the second argument, and zero if both arguments
     *         represent the same day.
     */
    public static int differenceInDays(Time later, Time earlier) {
        int offsetLater = DayDateImpl.offset((short) later.year(), (byte) later.month().toNum(), (byte) later.day());
        int offsetEarlier = DayDateImpl.offset((short) earlier.year(), (byte) earlier.month().toNum(), (byte) earlier.day());
        return offsetLater - offsetEarlier;
    }

    /**
     * Calculates the difference in whole days between the two given date. For
     * instance, the difference between 2011-06-04 and 2011-06-03 is one day.
     * The sign of this difference assumes that the first argument is later than
     * the second argument, if not, the calculated difference will be negative.
     * Two arguments representing the same day will have zero difference.
     * 
     * @param later The first time to compare.
     * @param earlier The second time to compare.
     * @return The difference in whole days between the two given dates. This
     *         is positive if the first argument represents a later day than
     *         the second argument, negative if the first argument represents an
     *         earlier day than the second argument, and zero if both arguments
     *         represent the same day.
     */
    public static int differenceInDays(DayDate later, DayDate earlier) {
        int offsetLater = DayDateImpl.offset((short) later.year(), (byte) later.month().toNum(), (byte) later.day());
        int offsetEarlier = DayDateImpl.offset((short) earlier.year(), (byte) earlier.month().toNum(), (byte) earlier.day());
        return offsetLater - offsetEarlier;
    }

}
