/*
 * Copyright 2011 Andreas Enblom
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.enblom.time;

/**
 * Factory for creating {@link Time} objects.
 * 
 * @author Andreas Enblom
 */
public interface TimeFactory {

    /**
     * @return The current time in the default timezone.
     */
    Time now();

    /**
     * @return The current time in the UTC timezone.
     */
    Time utc();

    /**
     * Returns a default time value: 1970-01-01 00:00:00.000 (the UNIX epoch
     * time). This is useful instead of <code>null</code> values.
     * 
     * @return The default time: 1970-01-01 00:00:00.000.
     */
    Time getDefault();

    /**
     * Creates a new {@link Time} instance representing the given time of day
     * of the given date.
     * 
     * @param date The date.
     * @param timeOfDay The time of day.
     * @return The new instance.
     */
    Time create(DayDate date, TimeOfDay timeOfDay);

    /**
     * Creates a new {@link Time} object with time of the given string
     * serialization, as returned by {@link Time#serialize()}
     * 
     * @param time The string serialization of the time, on the format
     *             <code>YYYYMMDDhhmmssnnn</code>, where <code>nnn</code> is the
     *             milliseconds.
     * @return The deserialized time object.
     * @throws IllegalArgumentException If the given string serialization does
     *                                  not represent a correct timestamp.
     */
    Time deserialize(String time) throws IllegalArgumentException;

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * <p>
     * The millisecond of the time is set to 0.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String year, String month, String day, String hour, String minute, String second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. out of range, null etc etc) - no exception is
     * ever thrown.
     * <p>
     * The millisecond of the time is set to 0.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(int year, int month, int day, int hour, int minute, int second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String year, String month, String day, String hour, String minute, String second, String millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. out of range, null etc etc) - no exception is
     * ever thrown.
     * 
     * @param year The year in the range 1000-9999.
     * @param month The month in the range 1-12.
     * @param day The day in the range 1-28, 1-29, 1-30 or 1-31 depending on
     *            month and year.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(int year, int month, int day, int hour, int minute, int second, int millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * <p>
     * The millisecond of the time is set to 0.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String date, String hour, String minute, String second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * <p>
     * The millisecond of the time is set to 0.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String date, int hour, int minute, int second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String date, String hour, String minute, String second, String millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String date, int hour, int minute, int second, int millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param date The date, represented as a string on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>.
     * @param timeOfDay The time of day.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(String date, TimeOfDay timeOfDay);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * <p>
     * The millisecond is set to 0.
     * 
     * @param date The date.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(DayDate date, String hour, String minute, String second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. out of range, null etc etc) - no exception is
     * ever thrown.
     * <p>
     * The millisecond is set to 0.
     * 
     * @param date The date.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(DayDate date, int hour, int minute, int second);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. not number, out of range, null etc etc) - no
     * exception is ever thrown.
     * 
     * @param date The date.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(DayDate date, String hour, String minute, String second, String millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * parameters. If the given parameters do not represent a correct time,
     * <code>null</code> is returned. This is the case even if the parameters
     * are totally wrong (i.e. out of range, null etc etc) - no exception is
     * ever thrown.
     * 
     * @param date The date.
     * @param hour The hour in the range 0-23.
     * @param minute The minute in the range 0-59.
     * @param second The second in the range 0-59.
     * @param millis The millisecond in the range 0-999.
     * @return The time object, or <code>null</code> if the provided parameters
     *         do not represent a correct time.
     */
    Time parse(DayDate date, int hour, int minute, int second, int millis);

    /**
     * Creates a new {@link Time} object, if possible, from the given
     * representation. If the given representation do not represent a correct
     * time, <code>null</code> is returned. This is the case even if the
     * representation is totally wrong - no exception is ever thrown.
     * 
     * @param time The timestamp, in the format <code>date time</code>, where
     *             <code>date</code> is on one of the formats
     *             <code>YYYYMMDD</code> and <code>YYYY-MM-DD</code>, and
     *             <code>time</code> is on one of the formats
     *             <code>hhmmss</code>, <code>hhmmssnnn</code>,
     *             <code>hh:mm:ss</code>, <code>hh:mm:ss.nnn</code>, where
     *             <code>nnn</code> is the millisecond.
     * @return The time object, or <code>null</code> if the provided paramater
     *         does not represent a correct time.
     */
    Time parse(String time);

    /**
     * Converts a {@link java.util.Date} to a {@link Time} object, using the
     * default time zone.
     * 
     * @param date The date to convert.
     * @return The converted time.
     */
    Time fromJavaUtilDate(java.util.Date date);

    /**
     * Converts a {@link java.util.Date} to a {@link Time} object, using the
     * given time zone.
     * 
     * @param date The date to convert.
     * @param timeZone The time zone, or <code>null</code> to use the default
     *                 time zone.
     * @return The converted time.
     */
    Time fromJavaUtilDate(java.util.Date date, java.util.TimeZone timeZone);

    /**
     * Converts a <code>long</code>, as returned from {@link Time#toLong()} to
     * a {@link Time} object.
     * 
     * @param time The <code>long</code> value.
     * @return The converted time.
     * 
     * @throws IllegalArgumentException If the given <code>long</code> value
     *                                  does not correctly represent a time.
     */
    Time fromLong(long time) throws IllegalArgumentException;


    /**
     * Converts a <code>long</code>, as returned from {@link Time#pack()} to a
     * {@link Time} object.
     * 
     * @param time The <code>long</code> value.
     * @return The converted time.
     * 
     * @throws IllegalArgumentException If the given <code>long</code> value
     *                                  does not correctly represent a time.
     */
    Time unpack(long time) throws IllegalArgumentException;

}
